package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.UUID;

/**
 * Browser action contains source url that redirects to certain Web page.
 * The browser action can be obtained via REST Client.
 * <p>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link BrowserAction.Builder}.
 */
public class BrowserAction extends AbstractModel implements IBrowserAction {

    private final UUID id;

    private final Proximity proximity;

    private final IDevice device;

    private final String url;

    private final int hashCode;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>
     * Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p>
     * <p>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Parcelable.Creator<BrowserAction> CREATOR = new Parcelable.Creator<BrowserAction>() {
        public BrowserAction createFromParcel(Parcel in) {
            final Builder builder = new Builder();
            Bundle bundle = in.readBundle(getClass().getClassLoader());

            if (bundle.getBoolean(PARCELABLE_HAS_BEACON)) {
                builder.setDevice(Beacon.CREATOR.createFromParcel(in));
            }

            return builder
                    .setId((UUID) bundle.getSerializable(Constants.ID))
                    .setProximity((Proximity) bundle.getSerializable(Constants.Action.PROXIMITY))
                    .setUrl(bundle.getString(Constants.Action.URL))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .build();
        }

        public BrowserAction[] newArray(int size) {
            return new BrowserAction[size];
        }
    };

    /**
     * Instantiates a new Browser action.
     *
     * @param builder BrowserAction Builder
     */
    private BrowserAction(Builder builder) {
        super(builder.databaseId);
        this.id = builder.id;
        this.proximity = builder.proximity;
        this.device = builder.beacon;
        this.url = builder.url;
        this.hashCode = HashCodeBuilder.init()
                .append(id)
                .append(proximity)
                .append(url)
                .append(device)
                .build();
    }

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public Proximity getProximity() {
        return proximity;
    }

    @Override
    public ActionType getType() {
        return ActionType.BROWSER;
    }

    @Override
    public IDevice getDevice() {
        return device;
    }

    @Override
    public String getUrl() {
        return url;
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (o == null || !(o instanceof BrowserAction)) {
            return false;
        }

        BrowserAction browserAction = (BrowserAction) o;

        return SDKEqualsBuilder.start()
                .equals(id, browserAction.id)
                .equals(proximity, browserAction.proximity)
                .equals(url, browserAction.url)
                .equals(device, browserAction.device)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.ID, id);
        bundle.putSerializable(Constants.Action.PROXIMITY, proximity);
        bundle.putParcelable(Constants.DEVICE, device);
        bundle.putString(Constants.Action.URL, url);
        bundle.putInt(Constants.DATABASE_ID, databaseId);

        final boolean shouldParcelBeacon = flags != PARCELABLE_FLAG_EXCLUDE_BEACON && device != null;

        bundle.putBoolean(PARCELABLE_HAS_BEACON, shouldParcelBeacon);

        dest.writeBundle(bundle);

        if (shouldParcelBeacon) {
            device.writeToParcel(dest, PARCELABLE_FLAG_EXCLUDE_ACTIONS);
        }
    }

    /**
     * The type Builder.
     */
    public static class Builder {
        private UUID id;
        private Proximity proximity;
        private IDevice beacon;
        private String url;
        private int databaseId;

        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return the database id
         */
        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets id.
         *
         * @param id the id
         * @return the id
         */
        public Builder setId(UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets one of three values: IMMEDIATE, NEAR, FAR.
         *
         * @param proximity the proximity
         * @return the proximity
         */
        public Builder setProximity(Proximity proximity) {
            this.proximity = proximity;
            return this;
        }

        /**
         * Sets device.
         *
         * @param beacon the device
         * @return the device
         */
        public Builder setDevice(IDevice beacon) {
            this.beacon = beacon;
            return this;
        }


        /**
         * Sets url.
         *
         * @param url the url
         * @return the url
         */
        public Builder setUrl(String url) {
            this.url = url;
            return this;
        }

        /**
         * Build browser action.
         *
         * @return the browser action
         */
        public BrowserAction build() {
            return new BrowserAction(this);
        }
    }

    /**
     * Creates BrowserAction from JSON Object
     *
     * @param jsonObject the json object
     * @return the browser action
     */
    public static BrowserAction from(final JSONObject jsonObject) {
        try {
            return new Builder()
                    .setProximity(JSONUtils.hasJSONKey(jsonObject, Constants.Action.PROXIMITY) ? Proximity.valueOf(jsonObject.getString(Constants.Action.PROXIMITY)) : null)
                    .setDevice(JSONUtils.hasJSONKey(jsonObject, Constants.DEVICE) ? Beacon.from(jsonObject.getJSONObject(Constants.DEVICE)) : null)
                    .setUrl(JSONUtils.getStringOrNull(jsonObject, Constants.Action.URL))
                    .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                    .build();
        } catch (JSONException e) {
            throw new IllegalArgumentException(e);
        }
    }
}
