package com.kontakt.sdk.android.common.model;


import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Represents Beacon basic values after resolve shuffled IBeacon
 */
public class BeaconId implements Parcelable {

    public static final Creator<BeaconId> CREATOR = new Creator<BeaconId>() {
        @Override
        public BeaconId createFromParcel(Parcel source) {
            return new BeaconId(source);
        }

        @Override
        public BeaconId[] newArray(int size) {
            return new BeaconId[size];
        }
    };

    private static final String PROXIMITY = "proximity";
    private static final String MAJOR = "major";
    private static final String MINOR = "minor";
    private static final String FUTURE_SHUFFLES = "futureId";
    private static final String IBEACON = "IBEACON";

    private UUID proximity;
    private int major;
    private int minor;

    private Integer hashCode = null;

    /**
     * Creates BeaconId with given proximity, major and minor
     *
     * @param proximity proximity uuid
     * @param major     major
     * @param minor     minor
     * @return BeaconId
     */
    public static BeaconId of(UUID proximity, int major, int minor) {
        return new BeaconId(proximity, major, minor);
    }

    private BeaconId(UUID proximity, int major, int minor) {
        this.proximity = proximity;
        this.major = major;
        this.minor = minor;
    }

    private BeaconId(Parcel source) {
        Bundle bundle = source.readBundle(getClass().getClassLoader());
        proximity = (UUID) bundle.getSerializable(PROXIMITY);
        major = bundle.getInt(MAJOR);
        minor = bundle.getInt(MINOR);
    }


    public int getMinor() {
        return minor;
    }

    public int getMajor() {
        return major;
    }

    public UUID getProximity() {
        return proximity;
    }

    //do not change implementation of toString() method, it is used as http request as well
    @Override
    public String toString() {
        return proximity.toString() + ":" + major + ":" + minor;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof BeaconId)) {
            return false;
        }

        if (o == this) {
            return true;
        }

        BeaconId beaconId = (BeaconId) o;
        return SDKEqualsBuilder.start()
                .equals(proximity, beaconId.proximity)
                .equals(major, beaconId.major)
                .equals(minor, beaconId.minor)
                .result();
    }

    @Override
    public int hashCode() {
        if (hashCode == null) {
            hashCode = HashCodeBuilder.init()
                    .append(proximity)
                    .append(major)
                    .append(minor)
                    .build();
        }
        return hashCode;
    }

    /**
     * Creates BeaconId from {@link IBeaconDevice}
     *
     * @param beaconDevice the beacon device
     * @return beaconId
     */
    public static BeaconId of(IBeaconDevice beaconDevice) {
        UUID proximity = beaconDevice.getProximityUUID();
        int major = beaconDevice.getMajor();
        int minor = beaconDevice.getMinor();
        return new BeaconId(proximity, major, minor);
    }

    /**
     * Creates Beacon id from string in json
     *
     * @param queriedBy string
     * @return BeaconId
     */
    public static BeaconId fromQueriedBy(String queriedBy) {
        if (queriedBy == null) {
            throw new NullPointerException("queriedBy is null");
        }

        String[] split = queriedBy.split(":");
        if (split.length != 3) {
            throw new IllegalArgumentException("queriedBy is not valid format");
        }

        UUID proximity = UUID.fromString(split[0]);
        int major = Integer.parseInt(split[1]);
        int minor = Integer.parseInt(split[2]);

        return new BeaconId(proximity, major, minor);
    }

    /**
     * Creates BeaconId from {@link JSONObject}
     *
     * @param jsonObject the json object
     * @return BeaconId
     */
    public static BeaconId fromJSONObject(JSONObject jsonObject) {
        UUID proximity = JSONUtils.getUUIDOrNull(jsonObject, PROXIMITY);
        if (proximity == null) {
            throw new IllegalStateException("proximity is not present in the response");
        }

        int major = JSONUtils.getInt(jsonObject, MAJOR, -1);
        if (major == -1) {
            throw new IllegalStateException("major is not present in the response");
        }

        int minor = JSONUtils.getInt(jsonObject, MINOR, -1);
        if (minor == -1) {
            throw new IllegalStateException("minor is not present in the response");
        }

        return new BeaconId(proximity, major, minor);
    }

    /**
     * Gets List of BeaconId from json object
     *
     * @param jsonObject the json object
     * @return list of beacon id
     */
    public static List<BeaconId> getIBeaconFutureIds(JSONObject jsonObject) {
        List<BeaconId> beaconIds = new ArrayList<BeaconId>();
        try {
            JSONObject futureShufflesObject = jsonObject.getJSONObject(FUTURE_SHUFFLES);
            JSONArray ibeaconFutureShuffles = futureShufflesObject.getJSONArray(IBEACON);

            for (int i = 0; i < ibeaconFutureShuffles.length(); i++) {
                JSONObject ibeaconFutureShufflesJSONObject = ibeaconFutureShuffles.getJSONObject(i);
                beaconIds.add(fromJSONObject(ibeaconFutureShufflesJSONObject));
            }
        } catch (JSONException e) {
            throw new IllegalStateException("futureIds is not present in the response");
        }
        return beaconIds;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(PROXIMITY, proximity);
        bundle.putInt(MAJOR, major);
        bundle.putInt(MINOR, minor);
        dest.writeBundle(bundle);
    }
}
