package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.JSONUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Beacon model represents JSON model of Beacon device with kontakt.io logic
 * applied.
 * <p/>
 * Beacon is a property of a Manager ({@link Manager}).
 * It acts also as a container for Actions and may be assigned to certain
 * Venue ({@link Venue}).
 * Beacon is obtainable via REST Client.
 * Please, note that there is also Beacon class representing real Beacon device that
 * appears immediately once it is detected by Android Device' BluetoothAdapter
 * ({@link android.bluetooth.BluetoothAdapter}).
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Beacon.Builder}.
 */
public class Beacon extends Device implements IBeacon {

    private final boolean shuffled;
    private final UUID secureProximityUUID;
    private final String secureNamespace;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>
     * Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p/>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Creator<Beacon> CREATOR = new Creator<Beacon>() {
        public Beacon createFromParcel(Parcel in) {
            final Bundle bundle = in.readBundle(getClass().getClassLoader());
            return new Beacon(bundle, in);
        }

        public Beacon[] newArray(int size) {
            return new Beacon[size];
        }
    };


    private Beacon(final Bundle bundle, final Parcel parcel) {
        super(bundle, parcel);
        this.shuffled = bundle.getBoolean(Constants.Devices.SHUFFLED);
        this.secureProximityUUID = (UUID) bundle.getSerializable(Constants.Devices.SECURE_PROXIMITY);
        this.secureNamespace = bundle.getString(Constants.Devices.SECURE_NAMESPACE);
    }

    /**
     * Instantiates a new Beacon model.
     *
     * @param builder Beacon builder
     */
    public Beacon(Builder builder) {
        super(builder);
        this.shuffled = builder.shuffled;
        this.secureProximityUUID = builder.secureProximityUUID;
        this.secureNamespace = builder.secureNamespace;
    }

    @Override
    public boolean isShuffled() {
        return shuffled;
    }

    @Override
    public UUID getSecureProximityUUID() {
        return secureProximityUUID;
    }

    @Override
    public String getSecureNamespace() {
        return secureNamespace;
    }

    @Override
    protected void parcelProperties(Bundle bundle) {
        bundle.putBoolean(Constants.Devices.SHUFFLED, shuffled);
        bundle.putSerializable(Constants.Devices.SECURE_PROXIMITY, secureProximityUUID);
        bundle.putString(Constants.Devices.SECURE_NAMESPACE, secureNamespace);
    }

    /**
     * The type Builder.
     */
    public static class Builder extends Device.Builder<Beacon, Builder> {

        private boolean shuffled = false;
        private UUID secureProximityUUID;
        private String secureNamespace;

        /**
         * Sets shuffled value
         *
         * @param shuffled boolean value
         * @return the builder
         */
        public Builder setShuffled(boolean shuffled) {
            this.shuffled = shuffled;
            return this;
        }

        /**
         * Sets secure proximity uuid
         *
         * @param secureProximityUUID uuid
         * @return the builder
         */
        public Builder setSecureProximityUUID(UUID secureProximityUUID) {
            this.secureProximityUUID = secureProximityUUID;
            return this;
        }

        /**
         * Sets secure namespace
         *
         * @param secureNamespace namespace
         * @return the builder
         */
        public Builder setSecureNamespace(String secureNamespace) {
            this.secureNamespace = secureNamespace;
            return this;
        }

        /**
         * Builds beacon
         *
         * @return beacon
         */
        public Beacon build() {
            return new Beacon(this);
        }
    }

    /**
     * Creates new instance of Beacon from JSON content. The raw content may be
     * obtained via REST Client.
     *
     * @param jsonObject the json object with content
     * @return the beacon instance
     */
    public static Beacon from(final JSONObject jsonObject) {
        final Access access = Access.valueOf(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.ACCESS));
        final Specification specification = Specification.valueOf(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.SPECIFICATION));
        final Model model = Model.valueOf(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.MODEL));
        final String hexedUrl = JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.URL);
        final DeviceType deviceType = DeviceType.valueOf(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEVICE_TYPE));
        final UUID secureProximityUUID = JSONUtils.getUUIDOrNull(jsonObject, Constants.Devices.SECURE_PROXIMITY);
        final String secureNamespace = JSONUtils.getStringOrNull(jsonObject, Constants.Devices.SECURE_NAMESPACE);
        try {
            JSONArray profiles = jsonObject.optJSONArray(Constants.Devices.PROFILES);

            Beacon.Builder builder = new Beacon.Builder();

            builder.setShuffled(JSONUtils.getBoolean(jsonObject, Constants.Devices.SHUFFLED, false))
                    .setSecureProximityUUID(secureProximityUUID)
                    .setSecureNamespace(secureNamespace)
                    .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                    .setProximityUUID(JSONUtils.getUUIDOrNull(jsonObject, Constants.Devices.PROXIMITY))
                    .setMajor(JSONUtils.getInt(jsonObject, Constants.Devices.MAJOR, 0))
                    .setMinor(JSONUtils.getInt(jsonObject, Constants.Devices.MINOR, 0))
                    .setMacAddress(JSONUtils.getString(jsonObject, Constants.Devices.MAC_ADDRESS, ""))
                    .setTxPower(JSONUtils.getInt(jsonObject, Constants.Devices.TX_POWER, -1))
                    .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.NAME))
                    .setAlias(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.ALIAS))
                    .setInterval(JSONUtils.getInt(jsonObject, Constants.Devices.INTERVAL, 0))
                    .setUniqueId(JSONUtils.getString(jsonObject, Constants.UNIQUE_ID, ""))
                    .setActionsCount(JSONUtils.getInt(jsonObject, Constants.Devices.ACTIONS_COUNT, 0))
                    .setManagerId(JSONUtils.getUUIDOrNull(jsonObject, Constants.Devices.MANAGER_ID))
                    .setVenue(JSONUtils.hasJSONKey(jsonObject, Constants.VENUE) ? Venue.from(jsonObject.getJSONObject(Constants.VENUE)) : null)
                    .setLatitude(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.LATITUDE))
                    .setLongitude(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.LONGITUDE))
                    .setAccess(access)
                    .setSpecification(specification)
                    .setModel(model)
                    .setFirmwareVersion(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.FIRMWARE_VERSION))
                    .setInstanceId(JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.INSTANCE_ID))
                    .setUrl(hexedUrl != null ? EddystoneUtils.fromHexedUrlToUrl(hexedUrl) : null)
                    .setNamespace(JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.NAMESPACE))
                    .setDeviceType(deviceType);

            if (profiles != null) {
                List<DeviceProfile> deviceProfileList = new ArrayList<DeviceProfile>();
                int profilesLength = profiles.length();
                for (int i = 0; i < profilesLength; i++) {
                    deviceProfileList.add(DeviceProfile.valueOf((String) profiles.get(i)));
                }
                builder.addDeviceProfiles(deviceProfileList);
            }

            return builder.build();

        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }


}
