package com.kontakt.sdk.android.ble.spec;

import java.util.UUID;

/**
 * IBeaconCharacteristic The Name enums depict specification given in
 * kontakt.io Beacon specification - version 2.
 * <p/>
 * More precisely each Name enum describes characteristic with its name,
 * Id, and order in service that it belongs to.
 *
 * @see <a href="docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon specification - version 2.</a>
 */
public enum KontaktDeviceCharacteristic implements GattCharacteristicModel {

    /**
     * The client characteristic configuration
     */
    CLIENT_CHARACTERISTIC_CONFIG("Client characteristic configuration", "00002902-0000-1000-8000-00805f9b34fb"),

    /**
     * The DEVICE_NAME characteristic provides Beacon name.
     * (Service 1, Characteristic 1)
     */
    DEVICE_NAME("Device Name",
            "00002a00-0000-1000-8000-00805f9b34fb"),
    /**
     * The APPEARANCE characteristic provides appearance of Beacon device.
     * (Service 1, Characteristic 2)
     */
    APPEARANCE("Appearance",
            "00002a01-0000-1000-8000-00805f9b34fb"),
    /**
     * The SLAVE_PREFERRED_CONNECTION_PARAMETERS provides connection
     * parameters in slave role.
     * (Service 1, Characteristic 3)
     */
    SLAVE_PREFERRED_CONNECTION_PARAMETERS("Slave Preferred Connection Parameters",
            "00002a04-0000-1000-8000-00805f9b34fb"),
    /**
     * The SERVICE_CHANGED.
     */
    SERVICE_CHANGED("Service Changed",
            "00002a05-0000-1000-8000-00805f9b34fb"),
    /**
     * The MANUFACTURER_NAME_STRING characteristic provides name of the manufacturer.
     */
    MANUFACTURER_NAME("Manufacturer Name",
            "00002a29-0000-1000-8000-00805f9b34fb"),
    /**
     * The FIRMWARE_REVISION_STRING characteristic provides access
     * firmware number.
     */
    FIRMWARE_REVISION_STRING("Firmware Revision String",
            "00002a26-0000-1000-8000-00805f9b34fb"),
    /**
     * The HARDWARE_REVISION_STRING characteristic provides access to
     * Beacon's hardware number.
     */
    HARDWARE_REVISION("Hardware Revision String",
            "00002a27-0000-1000-8000-00805f9b34fb"),
    /**
     * The TX_POWER_LEVEL characteristic provides access to Beacon's power
     * level.
     */
    TX_POWER_LEVEL("Tx Power Level",
            "00002a07-0000-1000-8000-00805f9b34fb"),
    /**
     * The BATTERY_LEVEL characteristic provides access to measured
     * battery level described in percentage value.
     */
    BATTERY_LEVEL("Battery Level",
            "00002a19-0000-1000-8000-00805f9b34fb"),
    /**
     * The PROXIMITY_UUID characteristic provides access to Beacon's proximity UUID.
     */
    PROXIMITY_UUID("Proximity UUID",
            "a1ea8111-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The MAJOR characteristic provides access to Beacon's Major value.
     */
    MAJOR("Major", "a1ea8112-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The MINOR characteristic provides access to Beacon's Minor value.
     */
    MINOR("Minor", "a1ea8113-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * Beacon Id characteristic.
     */
    BEACON_ID("Beacon id", "a1ea8115-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The PROPAGATED_DEVICE_NAME provides access to propagated beacon device name
     * which is seen in an applications.
     */
    PROPAGATED_DEVICE_NAME("Propagated Device Name",
            "a1ea8114-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The ADVERTISING_INTERVAL characteristic provides access to beacon
     * advertising interval.
     */
    ADVERTISING_INTERVAL("Advertising Interval",
            "a1ea8121-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The Master password characteristic.
     */
    MASTER_PASSWORD("Master password", "a1ea8137-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The PASSWORD characteristic provides access to beacon password.
     */
    PASSWORD("Password",
            "a1ea8131-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The SET_PASSWORD characteristic provides access set beacon password
     * during beacon authorization.
     */
    SET_PASSWORD("Set Password",
            "a1ea8132-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The RESET characteristic allows to reset beacon.
     */
    RESET("Reset",
            "a1ea8133-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The DEFAULT_SETTINGS characteristic allows to restore default
     * beacon settings.
     */
    DEFAULT_SETTINGS("Default Settings",
            "a1ea8134-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The BOOTLOADER characteristic allows to enable bootloader mode.
     */
    BOOTLOADER("Bootloader",
            "a1ea8135-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The NON_CONNECTABLE_MODE characteristic allows to beacon's transition
     * into non-connectable mode. In this mode beacon is propagating
     * advertising package for 20 minutes. Authorization with beacon is
     * not possible.
     * This characteristic is accessible from 2.6 firmware
     */
    NON_CONNECTABLE_MODE("Non-connectable mode",
            "a1ea8136-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * Allows to switch profile that the BLE device is transmitting.
     * Currently, there is possibility to switch between iBeacon and Eddystone
     * profiles.
     */
    ACTIVE_PROFILE("Active Profile",
            "a1ea813a-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The DFU_PACKET.
     */
    DFU_PACKET("DFU Packet",
            "00001532-1212-efde-1523-785feabcd123"),
    /**
     * The DFU_CONTROL_POINT.
     */
    DFU_CONTROL_POINT("DFU Control Point",
            "00001531-1212-efde-1523-785feabcd123"),
    /**
     * Namespace Id characteristic.
     */
    NAMESPACE_ID("Namespace id", "a1ea8117-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * Instance Id characteristic.
     */
    INSTANCE_ID("Instance id", "a1ea8118-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * URL Id characteristic.
     */
    URL_ID("URL id", "a1ea8119-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The Shuffle interval characteristic.
     */
    SHUFFLE_INTERVAL("Shuffle interval", "a1ea8138-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The Shuffle key characteristic.
     */
    SHUFFLE_KEY("Shuffle key", "a1ea8139-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The Sensor on characteristic.
     */
    SENSOR_ON("Sensor on", "a1ea8211-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The Accelerometer characteristic..
     */
    ACCELEROMETER("Accelerometer", "a1ea8212-0e1b-d4a1-b840-63f88c8da1ea"),
    /**
     * The secure write characteristic
     */
    SECURE_WRITE("Secure write", "a1ea81f1-0e1b-d4a1-b840-63f88c8da1ea"),

    /**
     * The secure write response characteristic
     */
    SECURE_RESPONSE("Secure response", "a1ea81f2-0e1b-d4a1-b840-63f88c8da1ea"),

    /**
     * The last processed token characteristic
     */
    SECURE_LAST_PROCESSED_REQUEST_TOKEN("Last access token", "a1ea81f3-0e1b-d4a1-b840-63f88c8da1ea"),

    /**
     * The secure control point characteristic
     */
    SECURE_CONTROL_POINT("Control point", "a1ea81f4-0e1b-d4a1-b840-63f88c8da1ea");


    private final String nameString;

    private final String id;

    /**
     * Instantiates a new Characteristic.
     *
     * @param nameString the name string
     * @param id         the id
     */
    KontaktDeviceCharacteristic(final String nameString, final String id) {
        this.nameString = nameString;
        this.id = id;
    }

    /**
     * Gets characteristic id.
     *
     * @return the id
     */
    public String getId() {
        return id;
    }

    @Override
    public String toString() {
        return nameString;
    }

    /**
     * Finds {@link KontaktDeviceCharacteristic} by uuid
     *
     * @param uuid
     * @return KontaktDeviceCharacteristic or null
     */
    public static KontaktDeviceCharacteristic valueOf(final UUID uuid) {
        final String characteristicId = uuid.toString();
        for (KontaktDeviceCharacteristic characteristic : values()) {
            if (characteristic.id.equals(characteristicId)) {
                return characteristic;
            }
        }
        return null;
    }
}