package com.kontakt.sdk.android.ble.service;

import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.common.log.Logger;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

class ScanController {
    static final ScanController NULL = new ScanController(new Builder()) {
        @Override
        void start() {

        }

        @Override
        void stop() {

        }
    };

    private static final int MESSAGE_DISMISS = -1;

    private static final int MESSAGE_MONITOR_START = 0;

    private static final int MESSAGE_MONITOR_STOP = 1;

    private final ForceScanScheduler forceScanScheduler;

    private final ScanPeriod scanPeriod;

    private final Runnable monitorActiveRunner;

    private final Runnable monitorPassiveRunner;

    private final BlockingQueue<Integer> messageQueue;

    private Thread producerThread;

    private Thread consumerThread;

    private ScanController(Builder builder) {
        this.scanPeriod = builder.scanPeriod;
        this.monitorActiveRunner = builder.monitorActiveRunner;
        this.monitorPassiveRunner = builder.monitorPassiveRunner;
        this.forceScanScheduler = builder.forceScanScheduler;
        this.messageQueue = new LinkedBlockingQueue<Integer>(1);
    }

    void start() {
        producerThread = new Producer(messageQueue,
                scanPeriod,
                "monitor-message-producer-thread");
        consumerThread = new Consumer(messageQueue,
                monitorActiveRunner,
                monitorPassiveRunner,
                forceScanScheduler,
                "monitor-message-consumer-thread");
        consumerThread.start();
        producerThread.start();
    }

    void stop() {
        if (producerThread != null) {
            producerThread.interrupt();
            producerThread = null;
            consumerThread = null;
        }
    }

    private static class Producer extends Thread implements Thread.UncaughtExceptionHandler {

        private final BlockingQueue<Integer> messageQueue;

        private final long monitorActivePeriod;

        private final long monitorPassivePeriod;

        private final boolean isInstantScanRequested;

        private Producer(BlockingQueue<Integer> messageQueue,
                         ScanPeriod scanPeriod,
                         String name) {
            super(name);
            this.messageQueue = messageQueue;
            this.monitorActivePeriod = scanPeriod.getActivePeriod();
            this.monitorPassivePeriod = scanPeriod.getPassivePeriod();

            this.isInstantScanRequested = monitorPassivePeriod == 0L;

            setUncaughtExceptionHandler(this);
        }

        @Override
        public void run() {
            try {
                boolean isFirstTimeScan = true;

                while (!Thread.currentThread().isInterrupted()) {
                    Logger.i("Starting monitoring");

                    if (isFirstTimeScan) {
                        messageQueue.put(MESSAGE_MONITOR_START);
                    } else if (!isInstantScanRequested) {
                        messageQueue.put(MESSAGE_MONITOR_START);
                    }

                    TimeUnit.MILLISECONDS.sleep(monitorActivePeriod);
                    Logger.i(": Stopping monitoring");

                    if (!isInstantScanRequested) {
                        messageQueue.put(MESSAGE_MONITOR_STOP);
                    }

                    TimeUnit.MILLISECONDS.sleep(monitorPassivePeriod);

                    isFirstTimeScan = false;
                }
                Logger.i("Dismissing consumer thread");

                if (isInstantScanRequested) {
                    messageQueue.put(MESSAGE_MONITOR_STOP);
                }

                messageQueue.put(MESSAGE_DISMISS);
            } catch (InterruptedException ignored) {
                try {
                    if (isInstantScanRequested) {
                        messageQueue.put(MESSAGE_MONITOR_STOP);
                    }

                    messageQueue.put(MESSAGE_DISMISS);
                } catch (InterruptedException e) {
                    Logger.i("Monitoring interrupted");
                }
            }
        }

        @Override
        public void uncaughtException(Thread thread, Throwable ex) {
            Logger.e(String.format("%s interrupted, exception swallowed.", thread.getName()));
        }
    }

    private static class Consumer extends Thread implements Thread.UncaughtExceptionHandler {

        private final BlockingQueue<Integer> messageQueue;
        private final Runnable monitorActiveRunner;
        private final Runnable monitorPassiveRunner;
        private final ForceScanScheduler forceScanScheduler;

        private Consumer(BlockingQueue<Integer> messageQueue,
                         Runnable monitorActiveRunner,
                         Runnable monitorPassiveRunner,
                         ForceScanScheduler forceScanScheduler,
                         String name) {
            super(name);
            this.messageQueue = messageQueue;
            this.monitorActiveRunner = monitorActiveRunner;
            this.monitorPassiveRunner = monitorPassiveRunner;
            this.forceScanScheduler = forceScanScheduler;

            setUncaughtExceptionHandler(this);
        }

        @Override
        public void run() {
            try {
                while (true) {
                    final int messageCode = messageQueue.take();
                    switch (messageCode) {
                        case MESSAGE_MONITOR_START:
                            monitorActiveRunner.run();
                            forceScanScheduler.start();
                            break;

                        case MESSAGE_MONITOR_STOP:
                            forceScanScheduler.stop();
                            monitorPassiveRunner.run();
                            break;
                        default:
                            throw new InterruptedException();
                    }
                }
            } catch (InterruptedException e) {
                Logger.i("Consumer thread interrupted");
            }
        }

        @Override
        public void uncaughtException(Thread thread, Throwable ex) {
            Logger.e(String.format("%s interrupted, exception swallowed.", thread.getName()));
        }
    }

    static class Builder {
        private Runnable monitorActiveRunner;
        private Runnable monitorPassiveRunner;
        private ScanPeriod scanPeriod;
        private ForceScanScheduler forceScanScheduler;

        public Builder setScanActiveRunner(Runnable monitorActiveRunner) {
            this.monitorActiveRunner = monitorActiveRunner;
            return this;
        }

        public Builder setScanPassiveRunner(Runnable monitorPassiveRunner) {
            this.monitorPassiveRunner = monitorPassiveRunner;
            return this;
        }

        public Builder setScanPeriod(ScanPeriod scanPeriod) {
            this.scanPeriod = scanPeriod;
            return this;
        }

        public Builder setForceScanScheduler(ForceScanScheduler forceScanScheduler) {
            this.forceScanScheduler = forceScanScheduler;
            return this;
        }

        public ScanController build() {
            return new ScanController(this);
        }
    }
}
