package com.kontakt.sdk.android.ble.service;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.os.Build;

import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.concurrent.TimeUnit;

final class Runners {

    private Runners() {
    }

    public static Runnable newRunner(final RunnerType runnerType,
                                     ScanConfiguration configuration) {
        SDKPreconditions.checkNotNull(runnerType, "Runner type cannot be null");
        SDKPreconditions.checkNotNull(configuration, "Configuration cannot be null");
        switch (runnerType) {
            case MONITOR_ACTIVE_RUNNER:
                return newMonitorActiveRunner(configuration);
            case MONITOR_PASSIVE_RUNNER:
                return newMonitorPassiveRunner(configuration);
            case FORCE_SCAN_RUNNER:
                return newForceScanRunner(configuration);
            default:
                throw new RuntimeException();
        }
    }

    private static Runnable newForceScanRunner(final ScanConfiguration configuration) {
        final BluetoothAdapter.LeScanCallback scanCallback = configuration.getScanCallback();
        final ForceScanConfiguration forceScanConfiguration = configuration.getScanContext().getForceScanConfiguration();
        final long forceScanActivePeriod = forceScanConfiguration.getForceScanActivePeriod();
        final long forceScanPassivePeriod = forceScanConfiguration.getForceScanPassivePeriod();

        return new Runnable() {
            @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
            @Override
            public void run() {
                final BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
                try {

                    if (bluetoothAdapter == null) {
                        Logger.e("Unexpected error occured - BluetoothAdapter is null.");
                        return;
                    }

                    while (!Thread.currentThread().isInterrupted()) {
                        Logger.d("Stopping Scan (force)");
                        bluetoothAdapter.stopLeScan(scanCallback);

                        Logger.d(String.format("Sleep during passive period: %s", String.valueOf(forceScanPassivePeriod)));
                        TimeUnit.MILLISECONDS.sleep(forceScanPassivePeriod);

                        Logger.d("Starting scan (force)");
                        bluetoothAdapter.startLeScan(scanCallback);

                        Logger.d(String.format("Sleep during active period: %s", String.valueOf(forceScanActivePeriod)));
                        TimeUnit.MILLISECONDS.sleep(forceScanActivePeriod);
                    }

                    bluetoothAdapter.stopLeScan(scanCallback);
                    Logger.d("Force scan finished");
                } catch (InterruptedException e) {
                    bluetoothAdapter.stopLeScan(scanCallback);
                    Logger.d("Force scan interrupted");
                }
            }
        };
    }

    private static Runnable newMonitorPassiveRunner(final ScanConfiguration configuration) {
        final MonitorCallback monitorCallback = (MonitorCallback) configuration.getScanCallback();

        return new Runnable() {

            @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
            @Override
            public void run() {
                monitorCallback.notifyScanStopped();

                BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
                adapter.stopLeScan(monitorCallback);

                monitorCallback.onScanStopped();
            }
        };
    }

    private static Runnable newMonitorActiveRunner(final ScanConfiguration serviceConfiguration) {
        final MonitorCallback monitorCallback = (MonitorCallback) serviceConfiguration.getScanCallback();

        return new Runnable() {
            @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
            @Override
            public void run() {

                final BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();

                if (bluetoothAdapter != null) {
                    bluetoothAdapter.startLeScan(monitorCallback);

                    monitorCallback.onScanStarted();

                    monitorCallback.notifyScanStarted();
                }
            }
        };
    }

    enum RunnerType {

        /**
         * The MONITOR_ACTIVE_RUNNER.
         */
        MONITOR_ACTIVE_RUNNER,

        /**
         * The MONITOR_PASSIVE_RUNNER.
         */
        MONITOR_PASSIVE_RUNNER,

        /**
         * The FORCE_SCAN_RUNNER.
         */
        FORCE_SCAN_RUNNER
    }
}
