package com.kontakt.sdk.android.ble.service;

import android.bluetooth.BluetoothDevice;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.EddystoneScanContext;
import com.kontakt.sdk.android.ble.configuration.scan.IBeaconScanContext;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneDiscoverer;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;

import java.util.Collection;
import java.util.LinkedList;

final class DefaultBluetoothDeviceDiscoverer implements BluetoothDeviceDiscoverer {

    private final Collection<BluetoothDeviceDiscoverer> discoverers;

    DefaultBluetoothDeviceDiscoverer(ScanContext scanContext,
                                     DiscoveryContract discoveryContract) {

        discoverers = new LinkedList<BluetoothDeviceDiscoverer>();

        ActivityCheckConfiguration activityCheckConfiguration = scanContext.getActivityCheckConfiguration();

        IBeaconScanContext iBeaconScanContext = scanContext.getIBeaconScanContext();

        if(iBeaconScanContext != null) {
            discoverers.add(new IBeaconDiscoverer(iBeaconScanContext, activityCheckConfiguration, discoveryContract));
        }

        EddystoneScanContext eddystoneScanContext = scanContext.getEddystoneScanContext();

        if(eddystoneScanContext != null) {
            discoverers.add(new EddystoneDiscoverer(eddystoneScanContext, activityCheckConfiguration, discoveryContract));
        }

    }

    @Override
    public boolean performDiscovery(BluetoothDevice bluetoothDevice, int rssi, byte[] scanResult) {
        for(BluetoothDeviceDiscoverer discoverer : discoverers) {
            if(discoverer.performDiscovery(bluetoothDevice, rssi, scanResult)) {
                return true;
            }
        }

        return false;
    }

    @Override
    public void clearResources() {
        for(BluetoothDeviceDiscoverer discoverer : discoverers) {
            discoverer.clearResources();
        }
    }

    @Override
    public void disable() {
        for(BluetoothDeviceDiscoverer discoverer : discoverers) {
            discoverer.disable();
        }
    }

    @Override
    public void reset() {
        for(BluetoothDeviceDiscoverer discoverer : discoverers) {
            discoverer.reset();
        }
    }

    @Override
    public synchronized void evictInactiveDevices() {
        for(BluetoothDeviceDiscoverer discoverer : discoverers) {
            discoverer.evictInactiveDevices();
        }
    }
}
