package com.kontakt.sdk.android.ble.filter.ibeacon;

import android.text.TextUtils;

import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconAdvertisingPacket;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.IBeaconPropertyValidator;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.UUID;

/**
 * Multi Filter allows to specify several parameters to search specific Beacon.
 */
public final class IBeaconMultiFilter implements IBeaconFilter {

    /**
     * The constant NO_MAJOR.
     */
    public static final int NO_MAJOR = 0;
    /**
     * The constant NO_MINOR.
     */
    public static final int NO_MINOR = 0;

    private final String deviceName;
    private final String address;
    private final UUID proximityUUID;
    private final String beaconUniqueId;
    private final String firmware;
    private final int major;
    private final int minor;

    public IBeaconMultiFilter(Builder builder) {
        this.deviceName = builder.deviceName;
        this.address = builder.address;
        this.proximityUUID = builder.proximityUUID;
        this.beaconUniqueId = builder.beaconUniqueId;
        this.firmware = builder.firmware;
        this.major = builder.major;
        this.minor = builder.minor;
    }

    @Override
    public boolean apply(IBeaconAdvertisingPacket target) {
        if (!TextUtils.isEmpty(deviceName) && !deviceName.equals(target.getName())) {
            return false;
        }

        if (!TextUtils.isEmpty(address) && !address.equals(target.getAddress())) {
            return false;
        }

        if (proximityUUID != null && !proximityUUID.equals(target.getProximityUUID())) {
            return false;
        }

        final String uniqueId = target.getBeaconUniqueId();

        if (!TextUtils.isEmpty(beaconUniqueId) && !TextUtils.isEmpty(uniqueId) && !beaconUniqueId.equals(uniqueId)) {
            return false;
        }

        if (! firmware.equals(target.getFirmwareVersion())) {
            return false;
        }

        if (minor != NO_MINOR && minor != target.getMinor()) {
            return false;
        }

        if (major != NO_MAJOR && major != target.getMajor()) {
            return false;
        }

        return true;
    }

    /**
     * Gets device name.
     *
     * @return the device name
     */
    public String getDeviceName() {
        return deviceName;
    }

    /**
     * Gets address.
     *
     * @return the address
     */
    public String getAddress() {
        return address;
    }

    /**
     * Gets proximity uUID.
     *
     * @return the proximity uUID
     */
    public UUID getProximityUUID() {
        return proximityUUID;
    }

    /**
     * Gets beacon unique id.
     *
     * @return the beacon unique id
     */
    public String getBeaconUniqueId() {
        return beaconUniqueId;
    }

    /**
     * Gets firmware.
     *
     * @return the firmware
     */
    public String getFirmwareVersion() {
        return firmware;
    }


    /**
     * Gets major.
     *
     * @return the major
     */
    public int getMajor() {
        return major;
    }

    /**
     * Gets minor.
     *
     * @return the minor
     */
    public int getMinor() {
        return minor;
    }

    /**
     * Multi filter Builder.
     */
    public static class Builder {
        private String deviceName = null;
        private String address = null;
        private UUID proximityUUID = null;
        private String beaconUniqueId = null;
        private String firmware = Constants.Devices.FIRMWARE_VERSION_NONE;
        private int major = NO_MAJOR;
        private int minor = NO_MINOR;

        Builder() { }

        /**
         * Sets device name.
         *
         * @param deviceName the device name
         * @return the builder instance
         */
        public Builder setDeviceName(final String deviceName) {
            SDKPreconditions.checkNotNull(deviceName, "Device name cannot be null.");
            this.deviceName = deviceName;
            return this;
        }

        /**
         * Sets address.
         *
         * @param address the address
         * @return the builder instance
         */
        public Builder setAddress(final String address) {
            SDKPreconditions.checkNotNull(address, "Address cannot be null.");
            this.address = address;
            return this;
        }

        /**
         * Sets proximity uUID.
         *
         * @param proximityUUID the proximity uUID
         * @return the builder instance
         */
        public Builder setProximityUUID(final UUID proximityUUID) {
            SDKPreconditions.checkNotNull(proximityUUID, "Proximity UUID cannot be null.");
            this.proximityUUID = proximityUUID;
            return this;
        }

        /**
         * Sets beacon unique id.
         *
         * @param beaconUniqueId the beacon unique id
         * @return the builder instance
         */
        public Builder setBeaconUniqueId(final String beaconUniqueId) {
            SDKPreconditions.checkNotNull(beaconUniqueId, "Beacon unique Id is null.");
            this.beaconUniqueId = beaconUniqueId;
            return this;
        }

        /**
         * Sets firmware.
         *
         * @param firmware the firmware
         * @return the builder instance
         */
        public Builder setFirmware(final String firmware) {
            SDKPreconditions.checkNotNull(firmware, "Firmware is null.");
            this.firmware = firmware;
            return this;
        }

        /**
         * Sets minor.
         *
         * @param minor the minor
         * @return the builder instance
         */
        public Builder setMinor(final int minor) {
            IBeaconPropertyValidator.validateMinor(minor);
            this.minor = minor;
            return this;
        }

        /**
         * Sets major.
         *
         * @param major the major
         * @return the builder instance
         */
        public Builder setMajor(final int major) {
            IBeaconPropertyValidator.validateMajor(minor);
            this.major = major;
            return this;
        }

        /**
         * Build multi filter.
         *
         * @return the multi filter
         */
        public IBeaconMultiFilter build() {
            return new IBeaconMultiFilter(this);
        }
    }
}