package com.kontakt.sdk.android.ble.filter.eddystone;

import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneUIDAdvertisingPacket;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneURLAdvertisingPacket;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * Factory class producing filters for Eddystone-specific packages: the UID packet, the URL packet, the TLM packet.
 */
public final class EddystoneFilters {

    private EddystoneFilters() {
    }

    /**
     * Creates new UID filter accepting UID packet with specified namespace id and instance id.
     *
     * @param namespaceId the namespace id
     * @param instanceId the instance id
     * @return the uID filter
     */
    public static UIDFilter newUIDFilter(final String namespaceId, final String instanceId) {
        SDKPreconditions.checkNotNull(namespaceId, "Namespace Id is null");
        SDKPreconditions.checkNotNull(instanceId, "Instance Id is null");
        return new UIDFilter() {
            @Override
            public boolean apply(EddystoneUIDAdvertisingPacket uidPacket) {
                return uidPacket.getNamespaceId().equals(namespaceId) &&
                        uidPacket.getInstanceId().equals(instanceId);
            }
        };
    }

    /**
     * Creates new URL filters accepting URL packet with specified url.
     *
     * @param url the url
     * @return the uRL filter
     */
    public static URLFilter newURLFilter(final String url) {
        SDKPreconditions.checkNotNull(url, "URL is null");

        return new URLFilter() {
            @Override

            public boolean apply(EddystoneURLAdvertisingPacket eddystonePacket) {
                return eddystonePacket.getUrl().equals(url);
            }
        };
    }

    /**
     * Creates new filter accepting UID packet with specified namespace id.
     *
     * @param namespaceId the namespace id
     * @return the uID filter
     */
    public static UIDFilter newNamespaceIdFilter(final String namespaceId) {
        SDKPreconditions.checkNotNull(namespaceId, "Namespace id is null");

        return new UIDFilter() {
            @Override
            public boolean apply(EddystoneUIDAdvertisingPacket eddystoneUIDAdvertisingPacket) {
                return eddystoneUIDAdvertisingPacket.getNamespaceId().equals(namespaceId);
            }
        };
    }

    /**
     * Creates new filter accepting UID packet with specified instance id.
     *
     * @param instanceId the instance id
     * @return the UID filter
     */
    public static UIDFilter newInstanceIdFilter(final String instanceId) {
        SDKPreconditions.checkNotNull(instanceId, "Instance id is null");

        return new UIDFilter() {
            @Override
            public boolean apply(EddystoneUIDAdvertisingPacket eddystoneUIDAdvertisingPacket) {
                return eddystoneUIDAdvertisingPacket.getInstanceId().equals(instanceId);
            }
        };
    }
}
