package com.kontakt.sdk.android.ble.discovery;

import android.util.SparseArray;

import com.kontakt.sdk.android.common.util.ConversionUtils;

import java.util.Arrays;

/**
 * Utility methods for bluetooth devices discovery.
 */
public final class DiscoveryUtils {

    private DiscoveryUtils() {
    }

    /**
     * Calculates distance from remote device.
     *
     * @param txPower the tx power
     * @param rssi the rssi
     * @return the double
     *
     * @see  <a href="http://stackoverflow.com/questions/20416218/understanding-ibeacon-distancing" target="_blank">Stack Overflow answer</a>
     */
    public static double calculateDistance(int txPower, double rssi) {
        if (rssi == 0) {
            return -1.0;
        }

        double ratio = rssi * 1.0 / txPower;
        if (ratio < 1.0) {
            return Math.pow(ratio, 10);
        } else {
            final double accuracy = (0.89976) * Math.pow(ratio, 7.7095) + 0.111;
            return accuracy;
        }
    }

    /**
     * Extracts data by type
     * @param type the type
     * @param scanRecord the scan record
     * @return array of bytes
     */
    public static byte[] extractDataByType(final int type, final byte[] scanRecord) {
        return extractMetaData(scanRecord).get(type);
    }

    /**
     * Reads scanResponse and advertising in block way
     * @param scanRecord
     * @return SparseArray of bytes
     */
    public static SparseArray<byte[]> extractMetaData(final byte[] scanRecord) {
        int index = 0;
        final SparseArray<byte[]> map = new SparseArray<byte[]>();
        final int scanRecordLength = scanRecord.length;
        while (index < scanRecordLength) {
            final int length = scanRecord[index++];

            if (length == 0) {
                break;
            }

            final int type = ConversionUtils.asInt(scanRecord[index]);

            if (type == 0) {
                break;
            }

            final byte[] data = Arrays.copyOfRange(scanRecord, index + 1, index + length);

            map.put(type, data);

            index += length;
        }

        return map;
    }

}
