package com.kontakt.sdk.android.ble.discovery;

import android.annotation.TargetApi;
import android.os.Build;
import android.util.SparseLongArray;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.ble.util.SafeSparseLongArray;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Abstraction for Bluetooth device discoverers in the SDK. The abstraction provides
 * convenient interface whenever you need to implement your own Bluetooth device discoverer.
 *
 * @param <Space> Space type parameter which the discovered devices belong to
 * @param <RBD>   Bluetooth device type parameter
 */
public abstract class AbstractBluetoothDeviceDiscoverer<Space, RBD extends RemoteBluetoothDevice> implements BluetoothDeviceDiscoverer {

    protected static boolean PROFILE_UNRECOGNIZED = false;

    protected static boolean PROFILE_RECOGNIZED_DEVICE_NOT_READY = true;

    protected static boolean PROFILE_RECOGNIZED_FILTERING_NOT_PASSED = true;

    protected static boolean PROFILE_RECOGNIZED_NO_BELONGING_SPACE_FOUND = true;

    private final ArrayList<RBD> EMPTY_DEVICE_LIST = new ArrayList<RBD>();

    private final DistanceComparator distanceComparator;

    private final DiscoveryContract discoveryContract;

    private final Collection<EventType> eventTypes;

    private final ActivityCheckConfiguration activityCheckConfiguration;

    private final Collection<Space> spaceSet;

    private final Map<Space, ReplacingArrayList<RBD>> spaceDeviceListMap = new ConcurrentHashMap<Space, ReplacingArrayList<RBD>>();

    private final SparseLongArray spaceTimestampArray;

    private final long devicesUpdateCallbackInterval;

    private long lastCallbackTime = 0;

    protected abstract BluetoothDeviceEvent createEvent(
            EventType eventType,
            Space space,
            ArrayList<RBD> deviceList);

    protected abstract void onBeforeDeviceLost(RBD device);

    protected AbstractBluetoothDeviceDiscoverer(DiscoveryContract discoveryContract,
                                                DistanceSort distanceSort,
                                                Collection<EventType> eventTypes,
                                                ActivityCheckConfiguration activityCheckConfiguration,
                                                Collection<Space> spaceSet,
                                                final long deviceUpdateCallbackInterval) {
        this.discoveryContract = discoveryContract;
        this.distanceComparator = new DistanceComparator(distanceSort);
        this.eventTypes = eventTypes;
        this.activityCheckConfiguration = activityCheckConfiguration;
        this.spaceSet = spaceSet;
        this.devicesUpdateCallbackInterval = deviceUpdateCallbackInterval;
        this.spaceTimestampArray = new SafeSparseLongArray(spaceSet.size());
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    @Override
    public void clearResources() {
        spaceTimestampArray.clear();
        spaceDeviceListMap.clear();
    }

    protected void sortIfEnabled(List<RBD> deviceList) {
        distanceComparator.sort(deviceList);
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    protected void notifySpacePresent(final int spaceHashcode, final long timestamp) {
        spaceTimestampArray.put(spaceHashcode, timestamp);
    }

    protected Collection<Space> getSpaceSet() {
        return spaceSet;
    }

    protected ReplacingArrayList<RBD> getDevicesInSpace(Space space) {
        return spaceDeviceListMap.get(space);
    }

    protected void insertDevicesIntoSpace(Space space, ReplacingArrayList<RBD> deviceList) {
        spaceDeviceListMap.put(space, deviceList);
    }

    protected void onSpaceEnteredEvent(Space space) {
        final EventType spaceEntered = EventType.SPACE_ENTERED;

        if (eventTypes.contains(spaceEntered)) {
            discoveryContract.onEvent(createEvent(
                    spaceEntered,
                    space,
                    EMPTY_DEVICE_LIST));
        }
    }

    protected void onDeviceDiscoveredEvent(Space space, RBD device) {
        final EventType deviceDiscovered = EventType.DEVICE_DISCOVERED;

        if (eventTypes.contains(deviceDiscovered)) {
            final ArrayList<RBD> deviceList = new ArrayList<RBD>(1);
            deviceList.add(device);

            discoveryContract.onEvent(createEvent(
                    EventType.DEVICE_DISCOVERED,
                    space,
                    deviceList
            ));
        }
    }

    protected void onDevicesUpdatedEvent(Space space, Collection<RBD> deviceList) {
        final long currentTimeMillis = System.currentTimeMillis();

        if (lastCallbackTime + devicesUpdateCallbackInterval > currentTimeMillis) {
            return;
        }

        final EventType devicesUpdate = EventType.DEVICES_UPDATE;

        if (eventTypes.contains(devicesUpdate)) {
            discoveryContract.onEvent(createEvent(
                    devicesUpdate,
                    space,
                    new ArrayList<RBD>(deviceList)));
        }

        lastCallbackTime = currentTimeMillis;
    }

    protected void onDeviceLostEvent(Space space, RBD device) {
        final EventType deviceLost = EventType.DEVICE_LOST;

        if (eventTypes.contains(deviceLost)) {
            ArrayList<RBD> deviceList = new ArrayList<RBD>(1);
            deviceList.add(device);

            discoveryContract.onEvent(createEvent(
                    deviceLost,
                    space,
                    deviceList
            ));
        }
    }

    protected void onSpaceAbandonedEvent(Space space) {

        final EventType eventType = EventType.SPACE_ABANDONED;

        if (eventTypes.contains(eventType)) {
            discoveryContract.onEvent(createEvent(
                    eventType,
                    space,
                    EMPTY_DEVICE_LIST));
        }
    }

    @Override
    public void evictInactiveDevices() {
        final long beaconInactivityTimeout = activityCheckConfiguration.getInactivityTimeout();
        for (Map.Entry<Space, ReplacingArrayList<RBD>> entry : spaceDeviceListMap.entrySet()) {

            final Space space = entry.getKey();
            Iterator<RBD> iBeaconIterator = entry.getValue().iterator();
            while (iBeaconIterator.hasNext()) {
                RBD remoteDevice = iBeaconIterator.next();
                long threshold = System.currentTimeMillis() - remoteDevice.getTimestamp();
                if (threshold > beaconInactivityTimeout) {
                    iBeaconIterator.remove();
                    onBeforeDeviceLost(remoteDevice);
                    onDeviceLostEvent(space, remoteDevice);
                }
            }
        }

        evictInactiveRegions();
    }

    @Override
    public void reset() {
        spaceDeviceListMap.clear();
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    private void evictInactiveRegions() {
        final Set<Space> spaces = new HashSet<Space>(spaceDeviceListMap.keySet());

        final long beaconInactivityTimeout = activityCheckConfiguration.getInactivityTimeout();

        for (final Space space : spaces) {
            final int regionHashCode = space.hashCode();
            final long regionTimestamp = spaceTimestampArray.get(regionHashCode, -1L);

            if (regionTimestamp != -1L) {
                final long timestampDifference = System.currentTimeMillis() - regionTimestamp;

                if (timestampDifference > beaconInactivityTimeout) {

                    spaceTimestampArray.get(regionHashCode, -1L);

                    spaceDeviceListMap.remove(space);

                    onSpaceAbandonedEvent(space);
                }
            }
        }
    }

    private static class DistanceComparator implements Comparator<RemoteBluetoothDevice> {

        private final int sortFactor;

        private DistanceComparator(DistanceSort distanceSort) {
            switch (distanceSort) {
                case DESC:
                    sortFactor = -1;
                    break;

                case ASC:
                    sortFactor = 1;
                    break;

                default:
                    sortFactor = 0;
            }
        }

        private boolean isEnabled() {
            return sortFactor != 0;
        }

        private void sort(final List<? extends RemoteBluetoothDevice> devices) {
            if (isEnabled()) {
                Collections.sort(devices, this);
            }
        }

        @Override
        public int compare(RemoteBluetoothDevice lhs, RemoteBluetoothDevice rhs) {
            int sortResult = (lhs.getDistance() < rhs.getDistance()) ? -1 : 1;
            return sortResult * sortFactor;
        }
    }
}
