package com.kontakt.sdk.android.ble.configuration.scan;

import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.filter.eddystone.TLMFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.UIDFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.URLFilter;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * Eddystone scan context provides describes criteria according to which the
 * Android device scans Eddystone devices. The context provides plenty of customizations
 * including:
 * searching by namespaces,
 * filters for UID, URL and TLM advertising packets,
 * selection of desired event types (for more information see {@link com.kontakt.sdk.android.ble.discovery.EventType}),
 * rssi signal manipulation,
 * sorting devices by their distance from Android device,
 * notifying events with updated device with specific interval
 */
public final class EddystoneScanContext extends AbstractProfileSpecificScanContext<IEddystoneNamespace> {

    /**
     * DEFAULT EddystoneScanContext provides .
     */
    public static final EddystoneScanContext DEFAULT = new EddystoneScanContext.Builder().build();

    private final Set<IEddystoneNamespace> eddystoneNamespaces;

    private final List<UIDFilter> uidFilters;

    private final List<URLFilter> urlFilters;

    private final List<TLMFilter> tlmFilters;

    private final Collection<EddystoneFrameType> triggerFrameTypes;

    private EddystoneScanContext(Builder builder) {
        super(builder);

        ScanContextValidator.validateNamespacesCount(builder.eddystoneNamespaces);

        ScanContextValidator.validateEddystoneUIDFiltersCount(builder.uidFilters);
        ScanContextValidator.validateEddystoneURLFiltersCount(builder.urlFilters);
        ScanContextValidator.validateEddystoneTLMFiltersCount(builder.tlmFilters);

        this.eddystoneNamespaces = Collections.unmodifiableSet(new HashSet<IEddystoneNamespace>(builder.eddystoneNamespaces));

        this.uidFilters = Collections.unmodifiableList(new LinkedList<UIDFilter>(builder.uidFilters));
        this.urlFilters = Collections.unmodifiableList(new LinkedList<URLFilter>(builder.urlFilters));
        this.tlmFilters = Collections.unmodifiableList(new LinkedList<TLMFilter>(builder.tlmFilters));

        this.triggerFrameTypes = Collections.unmodifiableCollection(builder.eddystoneFrameTypes);
    }

    /**
     * Provides UID advertising packet filters.
     *
     * @return the uID filters
     */
    public Collection<UIDFilter> getUIDFilters() {
        return uidFilters;
    }

    /**
     * Provides URL advertising packet filters.
     *
     * @return the uRL filters
     */
    public Collection<URLFilter> getURLFilters() {
        return urlFilters;
    }

    /**
     * Provides TLM advertising packet filters.
     *
     * @return the tLM filters
     */
    public Collection<TLMFilter> getTLMFilters() {
        return tlmFilters;
    }

    @Override
    public Collection<IEddystoneNamespace> getSpaces() {
        return eddystoneNamespaces;
    }

    /**
     * Gets trigger frame types.
     * The trigger frame types specify with which frame types received at least the Eddystone device
     * should be notified. The reuired frame type is {@link EddystoneFrameType#UID}. The all possible
     * combinations include:
     *
     * Eddystone device with {@link EddystoneFrameType#UID},
     * Eddystone device with {@link EddystoneFrameType#UID} and {@link EddystoneFrameType#URL},
     * Eddystone device with {@link EddystoneFrameType#UID} and {@link EddystoneFrameType#TLM},
     * Eddystone device with {@link EddystoneFrameType#UID}, {@link EddystoneFrameType#URL} and {@link EddystoneFrameType#TLM}.
     *
     * @return the trigger frame types
     */
    public Collection<EddystoneFrameType> getTriggerFrameTypes() {
        return triggerFrameTypes;
    }

    /**
     * Eddystone Scan context Builder.
     */
    public static final class Builder extends AbstractProfileSpecificScanContext.Builder<EddystoneScanContext, Builder> {

        private Set<IEddystoneNamespace> eddystoneNamespaces = new HashSet<IEddystoneNamespace>();

        private List<UIDFilter> uidFilters = new ArrayList<UIDFilter>();

        private List<URLFilter> urlFilters = new ArrayList<URLFilter>();

        private List<TLMFilter> tlmFilters = new ArrayList<TLMFilter>();

        private Collection<EddystoneFrameType> eddystoneFrameTypes = new HashSet<EddystoneFrameType>();

        @Override
        protected Builder getExtension() {
            return this;
        }

        @Override
        public EddystoneScanContext build() {
            if (eddystoneNamespaces.isEmpty()) {
                eddystoneNamespaces.add(EddystoneNamespace.EVERYWHERE);
            }

            if (!eddystoneFrameTypes.contains(EddystoneFrameType.UID)) {
                eddystoneFrameTypes.add(EddystoneFrameType.UID);
            }

            return new EddystoneScanContext(this);
        }

        /**
         * Sets filters accepting Eddystone devices with specific Eddystone UID frames.
         *
         * @param eddystoneFilters the UID eddystone filters
         * @return the builder instance
         */
        public Builder setUIDFilters(Collection<UIDFilter> eddystoneFilters) {
            SDKPreconditions.checkNotNull(eddystoneFilters, "Filters are null.");
            this.uidFilters.addAll(eddystoneFilters);
            return this;
        }

        /**
         * Sets filters accepting Eddystone devices with specific Eddystone TLM frames.
         *
         * @param eddystoneFilters the TLM eddystone filters
         * @return the builder instance
         */
        public Builder setTLMFilters(Collection<TLMFilter> eddystoneFilters) {
            SDKPreconditions.checkNotNull(eddystoneFilters, "Filters are null.");
            this.tlmFilters.addAll(eddystoneFilters);
            return this;
        }

        /**
         * Sets filters accepting Eddystone devices with specific Eddystone URL frames.
         *
         * @param eddystoneFilters the URL eddystone filters
         * @return the builder instance
         */
        public Builder setURLFilters(Collection<URLFilter> eddystoneFilters) {
            SDKPreconditions.checkNotNull(eddystoneFilters, "Filters are null.");
            this.urlFilters.addAll(eddystoneFilters);
            return this;
        }

        /**
         * Sets Eddystone namespaces within which Eddystone devices are searched.
         *
         * @param eddystoneNamespaces the eddystone namespaces
         * @return the builder instance
         */
        public Builder setEddystoneNamespaces(Collection<IEddystoneNamespace> eddystoneNamespaces) {
            SDKPreconditions.checkNotNull(eddystoneNamespaces, "Eddystone namespaces are null.");
            this.eddystoneNamespaces.addAll(eddystoneNamespaces);
            return this;
        }

        /**
         * Sets "trigger frame types". The trigger frame types specify with which eddystone-specific
         * frames at least the Eddystone device should be notified via
         * {@link com.kontakt.sdk.android.ble.manager.ProximityManager.ProximityListener}.
         * The Eddystone device is recognized as found when the Android device receives UID frame
         * at least.
         * If none trigger frame types are specified during scan context creation,
         * the {@link EddystoneFrameType#UID} is added by default.
         *
         * Possible combinations of "trigger frame types" are:
         * <ul>
         *     <li>{@link EddystoneFrameType#UID}</li>
         *     <li>{@link EddystoneFrameType#UID} and {@link EddystoneFrameType#URL}</li>
         *     <li>{@link EddystoneFrameType#UID} and {@link EddystoneFrameType#TLM}</li>
         *     <li>{@link EddystoneFrameType#UID}, {@link EddystoneFrameType#URL} and {@link EddystoneFrameType#TLM}</li>
         * </ul>
         *
         * @param eddystoneFrames the eddystone frames
         * @return the builder instance
         */
        public Builder setTriggerFrameTypes(Collection<EddystoneFrameType> eddystoneFrames) {
            SDKPreconditions.checkNotNull(eddystoneFrames, "Eddystone trigger frames");
            this.eddystoneFrameTypes.addAll(eddystoneFrames);
            return this;
        }

    }
}
