package com.kontakt.sdk.android.ble.configuration;

import com.kontakt.sdk.android.ble.manager.ProximityManager;

import java.util.concurrent.TimeUnit;

/**
 * Activity Check configuration describes the time boundary after which the beacon
 * device is interpreted as absent and than is removed from internal cache.
 *
 * The configuration consists of 2 parameters - removal inactivityTimeout
 * and checkPeriod.
 *
 * The former describes time threshold after which
 * beacon device is interpreted as inactive and should be removed.
 *
 * The latter specifies period of search of inactive beacons.
 */
public final class ActivityCheckConfiguration {

    /**
     * Predefined constant disabling device activity check configuration.
     */
    public static final ActivityCheckConfiguration DISABLED = new ActivityCheckConfiguration(-1, -1);

    /**
     * Predefined default Beacon Activity Check Configuration.
     * This configuration is set when no different is specified
     * during {@link ProximityManager}'s
     * configuration.
     * The DEFAULT configuration specifies beacon inactivity timeout equal to 5 seconds.
     * The
     */
    public static final ActivityCheckConfiguration MINIMAL = new ActivityCheckConfiguration(
            TimeUnit.SECONDS.toMillis(5),
            TimeUnit.SECONDS.toMillis(5));
    private final long inactivityTimeout;
    private final long checkPeriod;

    /**
     * Instantiates new immutable Beacon Activity Check Configuration instance
     *
     * @param inactivityTimeout the removal threshold [ms]
     * @param checkPeriod             the removal period [ms]
     */
    public ActivityCheckConfiguration(long inactivityTimeout,
                                      long checkPeriod) {
        this.inactivityTimeout = inactivityTimeout;
        this.checkPeriod = checkPeriod;
    }

    /**
     * Provides Beacon Inactivity Timeout [ms].
     *
     * @return the beacon inactivity timeout
     */
    public long getInactivityTimeout() {
        return inactivityTimeout;
    }

    /**
     * Provides check period.
     *
     * @return the check period [ms]
     */
    public long getCheckPeriod() {
        return checkPeriod;
    }
}
