package com.kontakt.sdk.android.ble.broadcast;

import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;

import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * Convenience Broadcast Receiver notifying about bluetooth state change. You
 * can make {@link com.kontakt.sdk.android.ble.manager.ProximityManager} scanning state
 *
 */
public class BluetoothStateChangeReceiver extends BroadcastReceiver {

    /**
     * Convenience constant equal to {@link BluetoothAdapter#ACTION_STATE_CHANGED}.
     * If you register {@link BluetoothStateChangeReceiver} with this action
     * set in {@link android.content.IntentFilter} you can leverage
     * {@link com.kontakt.sdk.android.ble.manager.ProximityManager}'s scan
     * states.
     */
    public static final String ACTION = BluetoothAdapter.ACTION_STATE_CHANGED;

    private final OnBluetoothStateChangeListener stateChangeListener;

    /**
     * Instantiates a new Bluetooth State Change Receiver.
     *
     * @param stateChangeListener the state change listener
     */
    public BluetoothStateChangeReceiver(OnBluetoothStateChangeListener stateChangeListener) {
        SDKPreconditions.checkNotNull(stateChangeListener,"State change listener cannot be null");
        this.stateChangeListener = stateChangeListener;
    }

    @Override
    public void onReceive(Context context, Intent intent) {

        if(! ACTION.equals(intent.getAction())) {
            return;
        }

        final int bluetoothState = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, -1);

        switch (bluetoothState) {

            case BluetoothAdapter.STATE_OFF:
                stateChangeListener.onBluetoothDisconnected();
                break;

            case BluetoothAdapter.STATE_ON:
                stateChangeListener.onBluetoothConnected();
                break;

            case BluetoothAdapter.STATE_TURNING_ON:
                stateChangeListener.onBluetoothConnecting();
                break;

            case BluetoothAdapter.STATE_TURNING_OFF:
                stateChangeListener.onBluetoothDisconnecting();
                break;
            default:
                break;
        }
    }
}
