package com.kontakt.sdk.android.http.interfaces;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.model.IVenue;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ETag;
import com.kontakt.sdk.android.http.HttpResult;
import com.kontakt.sdk.android.http.RequestDescription;
import com.kontakt.sdk.android.http.data.VenueData;
import com.kontakt.sdk.android.http.exception.ClientException;

import java.util.List;
import java.util.UUID;

/**
 * Venues API Accessor provides interface describing Venues management.
 */
public interface VenuesApiAccessor extends ApiAccessor {

    /**
     * Updates venue.
     *
     * Required parameter: venue Id. The optional parameters are: Venue Id, name, file, description. If any of the
     * optional values is null, it will not be updated.
     *
     * @param venueData the venue update data
     * @return the int
     * @throws ClientException if update fails
     */
    int updateVenue(VenueData venueData) throws ClientException;

    /**
     * Update venue.
     *
     * @param venueData the venue data
     * @param apiCallback the api callback
     */
    void updateVenue(VenueData venueData, UpdateApiCallback apiCallback);

    /**
     * Creates venue. Required parameter: name. The optional parameters are: fileData, description. If any of the optional
     * parameters is null, it will not be updated.
     *
     * @param venue the venue
     * @return the result
     * @throws ClientException is thrown if unexpected error has occured
     */
    HttpResult<IVenue> createVenue(VenueData venue) throws ClientException;

    /**
     * Create venue.
     *
     * @param venueData the venue data
     * @param apiCallback the api callback
     */
    void createVenue(VenueData venueData, ResultApiCallback<IVenue> apiCallback);

    /**
     * Requests venue by venue Id.
     * If the ETag's value matches the latest one
     * the code 304 (NOT MODIFIED) is returned.
     * Otherwise the method behaves identically as
     * {@link VenuesApiAccessor#getVenueImage(UUID)}
     *
     * @param venueId the venue id
     * @param eTag the e tag
     * @return the venue
     * @throws ClientException is thrown if unexpected error has occured
     */
    HttpResult<IVenue> getVenue(UUID venueId, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets venue.
     *
     * @param venueId the venue id
     * @param etag the etag
     * @param apiCallback the api callback
     */
    void getVenue(UUID venueId, SDKOptional<ETag> etag, ResultApiCallback<IVenue> apiCallback);

    /**
     * Deletes venue.
     *
     * @param venueId the venue Id
     * @return the Http Status. Successful deletion status is 200.
     * @throws ClientException is thrown if unexpected error has occured
     */
    int deleteVenue(UUID venueId) throws ClientException;

    /**
     * Delete venue.
     *
     * @param venueId the venue id
     * @param apiCallback the api callback
     */
    void deleteVenue(UUID venueId, UpdateApiCallback apiCallback);

    /**
     * Requests venue image by venue Id.
     *
     * @param venueId the venue id
     * @return the venue image
     * @throws ClientException is thrown if unexpected error has occured
     */
    HttpResult<FileData> getVenueImage(UUID venueId) throws ClientException;

    /**
     * Gets venue image.
     *
     * @param venueId the venue id
     * @param apiCallback the api callback
     */
    void getVenueImage(UUID venueId, ResultApiCallback<FileData> apiCallback);

    /**
     * Provides the list of venues. According to request description
     * the response:
     * <ul>
     *     <li>may contain sorted list items</li>
     *     <li>may contain empty body for conditional requests with ETag value included in headers</li>
     *     <li>may contain list items in ascending or descending order</li>
     *     <li>may contain ranged list items by specifying start index and offset</li>
     * </ul>
     *
     * @param requestDescription the request description
     * @return the devices
     * @throws ClientException the client exception
     */
    HttpResult<List<IVenue>> listVenues(RequestDescription requestDescription) throws ClientException;

    /**
     * List venues.
     *
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IVenue>> listVenues() throws ClientException;

    /**
     * List venues.
     *
     * @param requestDescription the request description
     * @param apiCallback the api callback
     */
    void listVenues(RequestDescription requestDescription, ResultApiCallback<List<IVenue>> apiCallback);

    /**
     * List venues.
     *
     * @param apiCallback the api callback
     */
    void listVenues(ResultApiCallback<List<IVenue>> apiCallback);
}
