package com.kontakt.sdk.android.http.interfaces;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.model.IAction;
import com.kontakt.sdk.android.common.model.IBrowserAction;
import com.kontakt.sdk.android.common.model.IContentAction;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ETag;
import com.kontakt.sdk.android.http.HttpResult;
import com.kontakt.sdk.android.http.data.ActionData;
import com.kontakt.sdk.android.http.exception.ClientException;

import java.io.File;
import java.util.List;
import java.util.UUID;

/**
 * Actions API accessor provides interface describing interaction with API and Actions management.
 */
public interface ActionsApiAccessor extends ApiAccessor {

    /**
     * @param actionData  {@link ActionData}
     * @param contentFile {@link File} file to upload
     * @return ContentAction
     * @throws ClientException
     */
    HttpResult<IContentAction> createContentAction(ActionData actionData, File contentFile) throws ClientException;

    /**
     * @param actionData  {@link ActionData}
     * @param contentFile {@link File} file to upload
     * @param apiCallback {@link ResultApiCallback}
     */
    void createContentAction(ActionData actionData, File contentFile, ResultApiCallback<IContentAction> apiCallback);

    /**
     * Create browser action.
     *
     * @param actionData the action
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<IBrowserAction> createBrowserAction(ActionData actionData) throws ClientException;

    /**
     * Create browser action.
     *
     * @param actionData        the action
     * @param resultApiCallback {@link ResultApiCallback}
     */
    void createBrowserAction(ActionData actionData, ResultApiCallback<IBrowserAction> resultApiCallback);

    /**
     * Requests action by Id sending ETag in newHeader additionally.
     * If the ETag's value matches the latest one
     * the code 304 (NOT MODIFIED) is returned.
     *
     * @param actionId the action id
     * @param eTag     the ETag optional value
     * @return the action
     * @throws ClientException the client exception
     */
    HttpResult<IAction> getAction(UUID actionId, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * @param actionId {@link UUID} of action
     * @return {@link IAction}
     * @throws ClientException
     */
    HttpResult<IAction> getAction(UUID actionId) throws ClientException;

    /**
     * @param actionId          {@link UUID} of action
     * @param etag              {@link ETag}
     * @param resultApiCallback {@link ResultApiCallback}
     */
    void getAction(UUID actionId, SDKOptional<ETag> etag, ResultApiCallback<IAction> resultApiCallback);

    /**
     * @param actionId          {@link UUID} of action
     * @param resultApiCallback {@link ResultApiCallback}
     */
    void getAction(UUID actionId, ResultApiCallback<IAction> resultApiCallback);

    /**
     * Updates Content action with new content of specified file.
     *
     * @param actionId the action id
     * @param file     the file
     * @return http status code
     * @throws ClientException is thrown if unexpected error has occured
     */
    int updateAction(UUID actionId, File file) throws ClientException;

    /**
     * @param actionId the action id
     * @param file     the file
     * @param callback {@link UpdateApiCallback}
     */
    void updateAction(UUID actionId, File file, UpdateApiCallback callback);

    /**
     * Updates url action
     *
     * @param actionId the action id
     * @param url      desired url
     * @return http status code
     * @throws ClientException
     */
    int updateAction(UUID actionId, String url) throws ClientException;

    /**
     * Updates url action
     *
     * @param actionId the action id
     * @param url      desired url
     * @param callback callback with http status code
     */
    void updateAction(UUID actionId, String url, UpdateApiCallback callback);

    /**
     * Deletes action.
     *
     * @param actionId the action id
     * @return the Http Status. Successful deletion should return Http Status: 200.
     * @throws ClientException is thrown if unexpected error has occured
     */
    int deleteAction(UUID actionId) throws ClientException;

    /**
     * @param actionId    the action id
     * @param apiCallback {@link UpdateApiCallback}
     */
    void deleteAction(UUID actionId, UpdateApiCallback apiCallback);

    /**
     * Requests action content.
     * If the ETag's value matches the latest one
     * the code 304 (NOT MODIFIED) is returned.
     *
     * @param actionId     the action id
     * @param eTagOptional the e tag optional
     * @return the action content
     * @throws ClientException is thrown if unexpected error has occured
     */
    HttpResult<FileData> getActionContent(UUID actionId, SDKOptional<ETag> eTagOptional) throws ClientException;

    /**
     * @param actionId          the action id
     * @param eTagSDKOptional   the e tag optional
     * @param resultApiCallback {@link UpdateApiCallback}
     */
    void getActionContent(UUID actionId, SDKOptional<ETag> eTagSDKOptional, ResultApiCallback<FileData> resultApiCallback);

    /**
     * Fetches actions for particular device
     *
     * @param deviceUniqueId unique id of beacon
     * @return list with actions, both Browser and Content
     * @throws ClientException is thrown if unexpected error has occured
     */
    HttpResult<List<IAction>> getActionsForDevice(String deviceUniqueId) throws ClientException;

    /**
     * Fetches actions for particular device
     *
     * @param deviceUniqueId    unique id of beacon
     * @param resultApiCallback the api-callback
     */
    void getActionsForDevice(String deviceUniqueId, ResultApiCallback<List<IAction>> resultApiCallback);
}
