package com.kontakt.sdk.android.http.data;

import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Abstract entity data represents data that are included to perform CRUD operation with
 * KontaktApiClient.
 */
public abstract class AbstractEntityData {

    private final EntityData entityData;

    /**
     * Instantiates a new Abstract entity data.
     *
     * @param entityData the entity data
     */
    protected AbstractEntityData(final EntityData entityData) {
        this.entityData = entityData;
    }

    /**
     * Provides parameter list.
     *
     * @return the parameter list
     */
    public final List<Map.Entry<String, String>> getParameters() {
        return entityData.getParameters();
    }

    /**
     * Checks if key exists in parameters
     *
     * @param key key to check
     * @return true if exists or false if not
     */
    public final boolean containsParameterKey(final String key) {
        SDKPreconditions.checkNotNull(key);

        for (Map.Entry<String, String> parameter : entityData.getParameters()) {
            if (parameter.getKey().equals(key)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Checks if key exists in parameters with provided value
     *
     * @param key   key to find
     * @param value desired value
     * @return true or false
     */
    public final boolean containsParameter(final String key, final String value) {
        SDKPreconditions.checkNotNull(key);
        SDKPreconditions.checkNotNull(value);

        for (Map.Entry<String, String> entry : entityData.getParameters()) {
            if (entry.getKey().equals(key) && entry.getValue().equals(value)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Provides parameter value.
     *
     * @param name the name
     * @return the parameter value or null if no argument exists with specified name
     */
    public final List<String> getParameterValues(final String name) {
        SDKPreconditions.checkNotNull(name);
        List<String> parameterValueList = new ArrayList<String>();
        for (Map.Entry<String, String> entry : entityData.getParameters()) {
            if (entry.getKey().equals(name)) {
                parameterValueList.add(entry.getValue());
            }
        }
        return Collections.unmodifiableList(parameterValueList);
    }
}
