package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.IAction;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import com.kontakt.sdk.android.http.data.ActionData;

import java.io.File;
import java.util.UUID;

/**
 * The type Http validator.
 */
final class RequestValidator {
    private RequestValidator() { }

    /**
     * Validate e tag.
     *
     * @param eTag the e tag
     */
    public static void validateETag(final SDKOptional<ETag> eTag) {
        SDKPreconditions.checkNotNull(eTag, "ETag optional cannot be null.");

        if(eTag.isPresent()) {
            SDKPreconditions.checkNotNullOrEmpty(eTag.get().getValue(), "ETag value is null.");
        }
    }

    static void contentActionCreationPreconditions(final ActionData actionData, final File file) {
        SDKPreconditions.checkState(actionData.containsParameter("actionType", IAction.ActionType.CONTENT.name()), "Action Type must be of CONTENT type");

        SDKPreconditions.checkState(actionData.containsParameterKey("uniqueId"), "No Parameter with Device Unique Id");

        SDKPreconditions.checkNotNull(file, "File data is null.");
        SDKPreconditions.checkState(file.exists(), "File does not exist");

        SDKPreconditions.checkState(actionData.containsParameterKey("proximity"), "Proximity not included as parameter. Should be one of three values: IMMEDIATE, NEAR, FAR");
    }

    static void browserActionCreationPreconditions(final ActionData actionData) {
        SDKPreconditions.checkState(actionData.containsParameter("actionType", IAction.ActionType.BROWSER.name()), "Action Type must be of BROWSER type");

        SDKPreconditions.checkState(actionData.containsParameterKey("uniqueId"), "No Parameter with Device Unique Id");

        SDKPreconditions.checkState(actionData.containsParameterKey("proximity"), "Proximity not included as parameter. Should be one of three values: IMMEDIATE, NEAR, FAR");

        SDKPreconditions.checkState(actionData.containsParameterKey("url"), "Url parameter must be included");
    }

    static void actionRetrievalPreconditions(final UUID actionId, final SDKOptional<ETag> etag) {
        SDKPreconditions.checkNotNull(actionId, "Action Id is null");
        validateETag(etag);
    }

    static void actionUpdatePreconditions(final UUID actionId, final File file) {
        SDKPreconditions.checkNotNull(actionId, "Content action ID is null.");
        SDKPreconditions.checkNotNull(file, "Content file to update is null.");
        SDKPreconditions.checkArgument(file.exists() && file.isFile(), "File either does not exist or is directory");
    }

    static void actionUpdatePreconditions(final UUID actionId, final String url) {
        SDKPreconditions.checkNotNull(actionId, "Content action ID is null.");
        SDKPreconditions.checkNotNullOrEmpty(url,"Url is null or empty");
    }

    static void actionDeletionPreconditions(final UUID actionId) {
        SDKPreconditions.checkNotNull(actionId, "Action id is null");
    }

    static void configsListingPreconditions(final RequestDescription requestDescription) {
        SDKPreconditions.checkState(requestDescription.containsParameter(Constants.Devices.DEVICE_TYPE, DeviceType.BEACON.name()),
                "DeviceType.BEACON is required.");
    }

    static void cloudConfigsListingPreconditions(final RequestDescription requestDescription) {
        SDKPreconditions.checkState(requestDescription.containsParameter(Constants.Devices.DEVICE_TYPE, DeviceType.CLOUD_BEACON.name()),
                "DeviceType.CLOUD_BEACON is required");
    }
}
