package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import com.squareup.okhttp.Headers;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.kontakt.sdk.android.common.util.HttpUtils.toUrlParameter;

/**
 * Class representing http call params, like startIndex and offset for paging, ETag header
 * <p>
 *     Example use:
 *      <pre>
 *      RequestDescription.Descriptor descriptor = RequestDescription.start()
                .setOffset(Integer.MAX_VALUE)
                .addParameter(Constants.Devices.DEVICE_TYPE, DeviceType.BEACON.name());

        RequestDescription description = descriptor.build();
 *      </pre>
 * </p>
 */
public class RequestDescription {

    private final int startIndex;
    private final int offset;
    private final ETag eTag;
    private final Order order;
    private final OrderBy orderBy;

    private final Headers headers;
    private final List<Map.Entry<String, String>> parameters;

    private RequestDescription(final Descriptor descriptor) {
        this.startIndex = descriptor.startIndex;
        this.offset = descriptor.offset;
        this.eTag = descriptor.eTag;
        this.order = descriptor.order;
        this.orderBy = descriptor.orderBy;
        this.headers = descriptor.additionalHeaders.build();
        this.parameters = descriptor.additionalParameters;
    }

    /**
     * Start descriptor.
     *
     * @return the descriptor
     */
    public static Descriptor start() {
        return new Descriptor();
    }

    /**
     * Extract header list.
     *
     * @return the list
     */
    public Headers getHeaders() {
        return headers;
    }

    /**
     * Extract parameters list.
     *
     * @return the list
     */
    public List<Map.Entry<String, String>> getParameters() {
        return parameters;
    }

    public boolean containsParameter(final String name, final String value) {
        SDKPreconditions.checkNotNull(name, "Parameter name is null");
        SDKPreconditions.checkNotNull(value, "Parameter value is null");

        for(Map.Entry<String, String> entry : parameters) {
            if(entry.getKey().equals(name) && entry.getValue().equals(value)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Gets e tag.
     *
     * @return the e tag
     */
    public ETag getETag() {
        return eTag;
    }

    /**
     * Gets order.
     *
     * @return the order
     */
    public Order getOrder() {
        return order;
    }

    /**
     * Gets order by.
     *
     * @return the order by
     */
    public OrderBy getOrderBy() {
        return orderBy;
    }

    /**
     * Gets results start index.
     *
     * @return the results start index
     */
    public int getResultsStartIndex() {
        return startIndex;
    }

    /**
     * Gets results offset.
     *
     * @return the results offset
     */
    public int getResultsOffset() {
        return offset;
    }

    /**
     * The type Descriptor.
     */
    public static final class Descriptor {

        private static final int FORBIDDEN_VALUE = -1;

        private Descriptor() { }

        private int startIndex = FORBIDDEN_VALUE;
        private int offset = FORBIDDEN_VALUE;
        private ETag eTag;
        private Order order;
        private OrderBy orderBy;

        private Headers.Builder additionalHeaders = new Headers.Builder();
        private List<Map.Entry<String, String>> additionalParameters = new ArrayList<Map.Entry<String, String>>();

        /**
         * Sets e tag.
         *
         * @param eTag the e tag
         * @return the e tag
         */
        public Descriptor setETag(final ETag eTag) {
            this.eTag = eTag;
            return this;
        }

        /**
         * Sets order.
         *
         * @param order the order
         * @return the order
         */
        public Descriptor setOrder(final Order order) {
            this.order = order;
            return this;
        }

        /**
         * Add header.
         *
         * @param name the name
         * @param value the value
         * @return the descriptor
         */
        public Descriptor addHeader(final String name, final String value) {
            this.additionalHeaders.add(name, value);
            return this;
        }

        /**
         * Add parameter.
         *
         * @param name the name
         * @param value the value
         * @return the descriptor
         */
        public Descriptor addParameter(final String name, final String value) {
            this.additionalParameters.add(toUrlParameter(name, value));
            return this;
        }

        /**
         * Add parameters.
         *
         * @param parameterList the parameter map
         * @return the descriptor
         */
        public Descriptor addParameters(final List<Map.Entry<String, String>> parameterList) {
            this.additionalParameters.addAll(parameterList);
            return this;
        }

        /**
         * Sets order by.
         *
         * @param orderBy the order by
         * @return the order by
         */
        public Descriptor setOrderBy(final OrderBy orderBy) {
            this.orderBy = orderBy;
            return this;
        }

        /**
         * Sets offset.
         *
         * @param offset the offset
         * @return the offset
         */
        public Descriptor setOffset(int offset) {
            this.offset = offset;
            return this;
        }

        /**
         * Sets start index. The start index must be higher or equal than 0.
         * Otherwise it is ignored.
         *
         * @param startIndex the start index
         * @return the start index
         */
        public Descriptor setStartIndex(int startIndex) {
            this.startIndex = startIndex;
            return this;
        }

        /**
         * Build request description.
         *
         * @return the request description
         */
        public RequestDescription build() {

            if(eTag != null) {
                additionalHeaders.add(eTag.getRequestName(), eTag.getValue());
            }

            if(order != null) {
                additionalParameters.add(toUrlParameter(Constants.SearchMeta.ORDER, order.name()));
            }

            if(orderBy != null) {
                additionalParameters.add(toUrlParameter(Constants.SearchMeta.ORDER_BY, orderBy.getArgument()));
            }

            if(startIndex >= 0) {
                additionalParameters.add(toUrlParameter(Constants.SearchMeta.START_INDEX, String.valueOf(startIndex)));
            }

            if(offset >= 0) {
                additionalParameters.add(toUrlParameter(Constants.SearchMeta.MAX_RESULT, String.valueOf(offset)));
            }

            return new RequestDescription(this);
        }
    }
}
