package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.model.IManager;
import com.kontakt.sdk.android.common.model.Manager;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.data.ManagerData;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.ManagersApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.kontakt.sdk.android.http.interfaces.UpdateApiCallback;

import org.json.JSONObject;

import java.util.List;
import java.util.Set;
import java.util.UUID;

import static com.kontakt.sdk.android.http.ApiConstants.Managers.MANAGERS_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Managers.MANAGER_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Managers.SUPERVISOR_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiMethods.Managers.MANAGERS_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Managers.MANAGER_ASSIGN;
import static com.kontakt.sdk.android.http.ApiMethods.Managers.MANAGER_CREATE;
import static com.kontakt.sdk.android.http.ApiMethods.Managers.MANAGER_DELETE;
import static com.kontakt.sdk.android.http.ApiMethods.Managers.MANAGER_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Managers.MANAGER_GET_SUBORDINATES;
import static com.kontakt.sdk.android.http.ApiMethods.Managers.MANAGER_UPDATE;

/**
 * {@link ManagersApiAccessor} implementation.
 */
final class ManagersApiAccessorImpl extends AbstractApiAccessor implements ManagersApiAccessor {

    /**
     * Instantiates a new Managers api client delegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    ManagersApiAccessorImpl(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
    }

    @Override
    public HttpResult<IManager> createManager(final ManagerData manager) throws ClientException {
        return createAndTransform(MANAGER_CREATE, manager, new SDKFunction<JSONObject, IManager>() {
            @Override
            public IManager apply(JSONObject object) {
                return Manager.from(object);
            }
        });
    }

    @Override
    public void createManager(ManagerData managerData, ResultApiCallback<IManager> apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameters(managerData.getParameters())
                .build();

        postAsyncAndBuildFromJSONObject(MANAGER_CREATE,
                requestDescription,
                HttpUtils.SC_CREATED,
                apiCallback,
                new SDKFunction<JSONObject, IManager>() {
                    @Override
                    public IManager apply(JSONObject object) {
                        return Manager.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IManager> getManager(UUID managerId, SDKOptional<ETag> eTagOptional) throws ClientException {
        String uri = String.format(MANAGER_GET, managerId.toString());

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTagOptional.isPresent() ? eTagOptional.get() : null)
                .build();
        return getAndTransform(uri, requestDescription, new SDKFunction<JSONObject, IManager>() {
            @Override
            public IManager apply(JSONObject object) {
                return Manager.from(object);
            }
        });
    }

    @Override
    public void getManager(UUID managerId, SDKOptional<ETag> etag, ResultApiCallback<IManager> apiCallback) {
        String uri = String.format(MANAGER_GET, managerId.toString());
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();

        getAsync(uri,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, IManager>() {
                    @Override
                    public IManager apply(JSONObject object) {
                        return Manager.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IManager> getManager(UUID managerId) throws ClientException {
        return getManager(managerId, SDKOptional.<ETag>absent());
    }

    @Override
    public void getManager(UUID managerId, ResultApiCallback<IManager> apiCallback) {
        getManager(managerId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<List<IManager>> listSubordinatesForManager(final UUID managerId, RequestDescription requestDescription) throws ClientException {
        final String uri = String.format(MANAGER_GET_SUBORDINATES, managerId.toString());

        return getAndTransformToList(uri, requestDescription, MANAGERS_PARAMETER, new SDKFunction<JSONObject, IManager>() {
            @Override
            public IManager apply(JSONObject object) {
                return Manager.from(object);
            }
        });
    }

    @Override
    public HttpResult<List<IManager>> listSubordinatesForManager(UUID managerId) throws ClientException {
        return listSubordinatesForManager(managerId, DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public void listSubordinatesForManager(UUID managerId,
                                           RequestDescription requestDescription,
                                           ResultApiCallback<List<IManager>> resultApiCallback) {
        final String uri = String.format(MANAGER_GET_SUBORDINATES, managerId.toString());

        transformToListAsynchronously(uri,
                requestDescription,
                HttpUtils.SC_OK,
                MANAGERS_PARAMETER,
                resultApiCallback,
                new SDKFunction<JSONObject, IManager>() {
                    @Override
                    public IManager apply(JSONObject object) {
                        return Manager.from(object);
                    }
                });
    }

    @Override
    public void listSubordinatesForManager(UUID managerId, ResultApiCallback<List<IManager>> resultApiCallback) {
        listSubordinatesForManager(managerId, DEFAULT_REQUEST_DESCRIPTION, resultApiCallback);
    }

    @Override
    public int updateManager(final ManagerData managerData) throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameters(managerData.getParameters())
                .build();

        return postAndReturnHttpStatus(MANAGER_UPDATE, requestDescription);
    }

    @Override
    public void updateManager(ManagerData managerData, UpdateApiCallback updateApiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameters(managerData.getParameters())
                .build();

        postAsyncAndReturnHttpStatus(MANAGER_UPDATE,
                requestDescription,
                updateApiCallback);
    }

    @Override
    public int assignManagersToSupervisor(UUID supervisorId, Set<UUID> managerIdSet) throws ClientException {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(SUPERVISOR_ID_PARAMETER, supervisorId.toString())
                .addParameters(HttpUtils.toUrlParameterList(MANAGER_ID_PARAMETER, managerIdSet))
                .build();

        return postAndReturnHttpStatus(MANAGER_ASSIGN, requestDescription);
    }

    @Override
    public void assignManagersToSupervisor(UUID supervisorId, Set<UUID> managerIdSet, UpdateApiCallback apiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(SUPERVISOR_ID_PARAMETER, supervisorId.toString())
                .addParameters(HttpUtils.toUrlParameterList(MANAGER_ID_PARAMETER, managerIdSet))
                .build();

        postAsyncAndReturnHttpStatus(MANAGER_ASSIGN, requestDescription, apiCallback);
    }

    @Override
    public HttpResult<List<IManager>> listManagers(RequestDescription requestDescription) throws ClientException {
        return getAndTransformToList(MANAGERS_GET, requestDescription, MANAGERS_PARAMETER, new SDKFunction<JSONObject, IManager>() {
            @Override
            public Manager apply(JSONObject object) {
                return Manager.from(object);
            }
        });
    }

    @Override
    public HttpResult<List<IManager>> listManagers() throws ClientException {
        return listManagers(DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public void listManagers(RequestDescription requestDescription, ResultApiCallback<List<IManager>> apiCallback) {
        transformToListAsynchronously(MANAGERS_GET,
                requestDescription,
                HttpUtils.SC_OK,
                MANAGERS_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, IManager>() {
                    @Override
                    public IManager apply(JSONObject object) {
                        return Manager.from(object);
                    }
                });
    }

    @Override
    public void listManagers(ResultApiCallback<List<IManager>> apiCallback) {
        listManagers(DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public int deleteManager(UUID managerId) throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(MANAGER_ID_PARAMETER, managerId.toString())
                .build();

        return postAndReturnHttpStatus(MANAGER_DELETE, requestDescription);
    }

    @Override
    public void deleteManager(UUID managerId, UpdateApiCallback apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(MANAGER_ID_PARAMETER, managerId.toString())
                .build();

        postAsyncAndReturnHttpStatus(MANAGER_DELETE, requestDescription, apiCallback);
    }
}
