package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.util.Closeables;
import com.kontakt.sdk.android.http.interfaces.ActionsApiAccessor;
import com.kontakt.sdk.android.http.interfaces.CommandApiAccessor;
import com.kontakt.sdk.android.http.interfaces.CommonApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ConfigurationApiAccessor;
import com.kontakt.sdk.android.http.interfaces.DevicesApiAccessor;
import com.kontakt.sdk.android.http.interfaces.FirmwareApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ManagersApiAccessor;
import com.kontakt.sdk.android.http.interfaces.VenuesApiAccessor;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.HashMap;

/**
 * API client provides interaction with kontakt.io REST API.
 *
 * @see <a href="http://docs.kontakt.io/rest-api/quickstart/">Kontakt.io REST API Quick Start Guide</a>
 */
public class KontaktApiClient extends AbstractKontaktApiClient {
    private final HashMap<String, Object> apiAccessorMap;

    public static final int ACCEPT_VERSION = AbstractKontaktApiClient.ACCEPT_VERSION;

    public KontaktApiClient() {
        this(KontaktSDK.getInstance().getApiKey(), API_URL);
    }

    protected KontaktApiClient(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
        apiAccessorMap = new HashMap<String, Object>();
    }

    @Override
    /**
     * {@inheritDoc}
     */
    protected DevicesApiAccessor devicesApi() {
        return putIfAbsentAndReturnApiAccessor(DevicesApiAccessorImpl.class, "beaconsApi");
    }

    @Override
    /**
     * {@inheritDoc}
     */
    protected VenuesApiAccessor venuesApi() {
        return putIfAbsentAndReturnApiAccessor(VenuesApiAccessorImpl.class, "venuesApi");
    }

    @Override
    /**
     * {@inheritDoc}
     */
    protected ManagersApiAccessor managersApi() {
        return putIfAbsentAndReturnApiAccessor(ManagersApiAccessorImpl.class, "managersApi");
    }

    @Override
    /**
     * {@inheritDoc}
     */
    protected ConfigurationApiAccessor configurationApi() {
        return putIfAbsentAndReturnApiAccessor(ConfigurationApiAccessorImpl.class, "configurationApi");
    }

    @Override
    /**
     * {@inheritDoc}
     */
    protected FirmwareApiAccessor firmwareApi() {
        return putIfAbsentAndReturnApiAccessor(FirmwareApiAccessorImpl.class, "firmwareApi");
    }

    @Override
    /**
     * {@inheritDoc}
     */
    protected ActionsApiAccessor actionsApi() {
        return putIfAbsentAndReturnApiAccessor(ActionsApiAccessorImpl.class, "actionsApi");
    }

    @Override
    protected CommonApiAccessor commonsApi() {
        return putIfAbsentAndReturnApiAccessor(CommonApiAccessorImpl.class, "commonApi");
    }

    @Override
    protected CommandApiAccessor commandApi() {
        return putIfAbsentAndReturnApiAccessor(CommandApiAccessorImpl.class, "commandApi");
    }

    @Override
    /**
     * {@inheritDoc}
     */
    public void close() {
        synchronized (apiAccessorMap) {
            for (Object apiAccessor : apiAccessorMap.values()) {
                try {
                    Closeables.close((AbstractApiAccessor) apiAccessor, true);
                } catch (IOException ignored) {
                }
            }

            apiAccessorMap.clear();
        }
    }

    @SuppressWarnings("unchecked")
    private <T> T putIfAbsentAndReturnApiAccessor(final Class<T> clazz, final String accessorKey) {
        synchronized (apiAccessorMap) {
            T accessorInstance = (T) apiAccessorMap.get(accessorKey);
            if (accessorInstance == null) {
                try {
                    final Constructor<T> constructor = clazz.getDeclaredConstructor(String.class, String.class);
                    accessorInstance = constructor.newInstance(apiKey, apiUrl);
                    apiAccessorMap.put(accessorKey, accessorInstance);
                } catch (Exception e) {
                    throw new IllegalStateException(e);
                }
            }

            return accessorInstance;
        }
    }

}
