package com.kontakt.sdk.android.http;

import android.util.SparseArray;

import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * Result contains data loaded from Http Response when interaction with REST Client is desired.
 *
 * @param <T> the generic type parameter
 */
public final class HttpResult<T> {

    private static final SparseArray<String> API_STATUS_INFO_CACHE;

    static {
        API_STATUS_INFO_CACHE = new SparseArray<String>();

        API_STATUS_INFO_CACHE.put(HttpUtils.SC_OK, "Request has been valid and successful");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_CREATED, "Entity has been created successfully");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_NO_CONTENT,"Request has been valid and there is no content");

        API_STATUS_INFO_CACHE.put(HttpUtils.SC_SEE_OTHER, "Request has been valid, but needed to be redirected elsewhere");

        API_STATUS_INFO_CACHE.put(HttpUtils.SC_BAD_REQUEST, "Request contains invalid values or is in invalid format");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_UNAUTHORIZED, "Unauthorized access. Most likely Api-Key hasn’t been sent or empty");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_FORBIDDEN, "The request was a valid request, but the server is refusing to respond to it");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_NOT_FOUND, "Resource not found");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_METHOD_NOT_ALLOWED, "A request was made of a resource using a request method not supported by that resource; for example, using GET on a form which requires data to be presented via POST, or using PUT on a read-only resource.");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_CONFLICT, "Request could not be processed because of conflict");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_UNSUPPORTED_MEDIA_TYPE, "Version or Mediatype not found");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_UNPROCESSABLE_ENTITY, "Parameters validation error");

        API_STATUS_INFO_CACHE.put(HttpUtils.SC_INTERNAL_SERVER_ERROR, "Internal Server Error occured");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_BAD_GATEWAY, "The server was acting as a gateway or proxy and received an invalid response from the upstream server.");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_GATEWAY_TIMEOUT, "The server is currently unavailable (because it is overloaded or down for maintenance). Generally, this is a temporary state");
        API_STATUS_INFO_CACHE.put(HttpUtils.SC_NOT_MODIFIED, "The resource has not been modified since the version specified by the request newHeader If-None-Match.");
    }

    private final T result;

    private final int httpStatus;

    private final String message;

    private final ETag eTag;

    private final SearchMeta searchMeta;

    private final String reasonPhrase;

    private HttpResult(final Builder<T> builder) {
        this.result = builder.value;
        this.httpStatus = builder.httpStatus;
        final String m = API_STATUS_INFO_CACHE.get(httpStatus);
        this.message = m == null ? "" : m;
        this.eTag = ETag.of(builder.etagValue);
        this.searchMeta = builder.searchMeta;
        this.reasonPhrase = builder.reasonPhrase;
    }

    /**
     * Gets value.
     *
     * @return the value
     */
    public T get() {
        SDKPreconditions.checkState(isPresent(), String.format("Invalid http status: %d (%s). Result is null.",
                httpStatus,
                API_STATUS_INFO_CACHE.get(httpStatus)));
        return result;
    }

    /**
     * Gets HttpResponse's status code.
     *
     * @return the status code
     */
    public int getStatusCode() {
        return httpStatus;
    }

    /**
     * Checkes whether result value is present.
     *
     * @return the boolean
     */
    public boolean isPresent() {
        return result != null;
    }

    /**
     * Gets message.
     *
     * @return the message
     */
    public String getMessage() {
        return message;
    }

    /**
     * Checks whether ETag is present.
     *
     * @return the boolean
     */
    public boolean isETagPresent() {
        return eTag != null;
    }

    /**
     * Gets ETag.
     *
     * @return the e tag
     * @see <a href="http://www.restpatterns.org/HTTP_Headers/If-Modified-Since" target="_blank">ETag - explanation.</a>
     * @see <a href="http://en.wikipedia.org/wiki/HTTP_ETag#Typical_usage" target="_blank">ETag - Wikipedia's explanation.</a>
     */
    public ETag getETag() {
        SDKPreconditions.checkNotNull(eTag, "ETag is not present");

        return eTag;
    }

    /**
     * Is search meta present.
     *
     * @return the boolean
     */
    public boolean isSearchMetaPresent() {
        return searchMeta != null;
    }

    /**
     * Gets search meta.
     *
     * @return the search meta
     */
    public SearchMeta getSearchMeta() {
        SDKPreconditions.checkNotNull(searchMeta, "SearchMeta is not present");

        return searchMeta;
    }

    /**
     * Gets reason phrase.
     *
     * @return the reason phrase
     */
    public String getReasonPhrase() {
        return reasonPhrase;
    }

    /**
     * Of http result.
     *
     * @param <T>        the type parameter
     * @param httpStatus the http status
     * @return the http result
     */
    public static <T> HttpResult<T> of(final int httpStatus) {
        return new Builder<T>()
                .setHttpStatus(httpStatus)
                .build();
    }

    /**
     * The type Builder.
     *
     * @param <T> the type parameter
     */
    public static final class Builder<T> {
        private T value;
        private int httpStatus;
        private String etagValue;
        private SearchMeta searchMeta;
        private String reasonPhrase;

        /**
         * Sets value.
         *
         * @param value the value
         * @return the builder
         */
        public Builder<T> setValue(T value) {
            this.value = value;
            return this;
        }

        /**
         * Sets http status.
         *
         * @param httpStatus the http status
         * @return the builder
         */
        public Builder<T> setHttpStatus(int httpStatus) {
            this.httpStatus = httpStatus;
            return this;
        }

        /**
         * Sets ETag value.
         *
         * @param etagValue the etag value
         * @return the builder
         */
        public Builder<T> setETagValue(String etagValue) {
            this.etagValue = etagValue;
            return this;
        }

        /**
         * Sets search meta.
         *
         * @param searchMeta the search meta
         * @return the search meta
         */
        public Builder<T> setSearchMeta(final SearchMeta searchMeta) {
            this.searchMeta = searchMeta;
            return this;
        }

        /**
         * Sets reason phrase.
         *
         * @param reasonPhrase the reason phrase
         * @return the reason phrase
         */
        public Builder<T> setReasonPhrase(final String reasonPhrase) {
            this.reasonPhrase = reasonPhrase;
            return this;
        }

        /**
         * Builds http result.
         *
         * @return the http result
         */
        public HttpResult<T> build() {
            return new HttpResult<T>(this);
        }
    }
}
