package com.kontakt.sdk.android.http;

import android.text.TextUtils;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.interfaces.SDKThrowableFunction;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import com.squareup.okhttp.Response;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * The type Fluent response.
 *
 * @param <T> the type parameter
 */
public final class FluentResponse<T> {

    private final Response httpResponse;
    private final int actualHttpStatus;
    private final List<Integer> expectedHttpStatuses;

    /**
     * Of fluent response.
     *
     * @param <T>                the type parameter
     * @param httpResponse       the http response
     * @param expectedHttpStatus the expected http status
     * @return the fluent response
     */
    public static <T> FluentResponse<T> of(final Response httpResponse, final int expectedHttpStatus) {
        SDKPreconditions.checkNotNull(httpResponse, "Http Response is null.");
        return new FluentResponse<T>(httpResponse, new int[]{expectedHttpStatus});
    }

    /**
     * Creates fluent responses with {@link Response} and array of expected http codes
     * <p>
     * If returned http status code is in expected array all transformation will be done, otherwise not
     * </p>
     *
     * @param httpResponse         the http response
     * @param expectedHttpStatuses the expected statuses
     * @return the fluent response
     */
    public static <T> FluentResponse<T> of(final Response httpResponse, final int[] expectedHttpStatuses) {
        SDKPreconditions.checkNotNull(httpResponse, "Http Response is null.");
        return new FluentResponse<T>(httpResponse, expectedHttpStatuses);
    }

    private FluentResponse(final Response httpResponse, final int[] expectedHttpStatus) {
        this.httpResponse = httpResponse;
        this.actualHttpStatus = httpResponse.code();
        this.expectedHttpStatuses = new ArrayList<Integer>();
        for (int expectedStatus : expectedHttpStatus) {
            expectedHttpStatuses.add(expectedStatus);
        }
    }


    private boolean isInExpectedHttpStatuses() {
        return expectedHttpStatuses.contains(actualHttpStatus);
    }

    /**
     * Transform to jSON.
     *
     * @return the jSON response
     */
    public JSONResponse<T> transformToJSON() {
        try {
            if (isInExpectedHttpStatuses()) {
                String response = httpResponse.body().string();
                if (!TextUtils.isEmpty(response)) {
                    return new JSONResponse<T>(SDKOptional.of(new JSONObject(response)));
                }
            }
            return new JSONResponse<T>(SDKOptional.<JSONObject>absent());
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Transform to byte array.
     *
     * @return the byte array response
     */
    public ByteArrayResponse<T> transformToByteArray() {
        try {
            return new ByteArrayResponse<T>(httpResponse.body().bytes());
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * The type JSON response.
     *
     * @param <T> the type parameter
     */
    public final class JSONResponse<T> {

        private final SDKOptional<JSONObject> jsonObjectOptional;

        private JSONResponse(final SDKOptional<JSONObject> jsonObject) {
            this.jsonObjectOptional = jsonObject;
        }

        /**
         * To single entity.
         *
         * @param buildFunction the build function
         * @return the http result
         */
        public HttpResult<T> toSingleEntity(final SDKFunction<JSONObject, T> buildFunction) {

            final HttpResult.Builder<T> resultBuilder = new HttpResult.Builder<T>()
                    .setHttpStatus(actualHttpStatus)
                    .setReasonPhrase(httpResponse.message());

            if (!jsonObjectOptional.isPresent()) {
                return resultBuilder.build();
            }

            final JSONObject jsonObject = jsonObjectOptional.get();

            resultBuilder.setSearchMeta(JSONUtils.transformOrReturnNull(jsonObject, SearchMeta.JSON_ENTRY, new SDKFunction<JSONObject, SearchMeta>() {
                @Override
                public SearchMeta apply(JSONObject object) {
                    try {
                        return SearchMeta.from(jsonObject.getJSONObject(SearchMeta.JSON_ENTRY));
                    } catch (JSONException e) {
                        throw new IllegalStateException(e);
                    }
                }
            }))
                    .setETagValue(httpResponse.header(ETag.HEADER_NAME_RESPONSE))
                    .setHttpStatus(actualHttpStatus)
                    .setSearchMeta(parseSearchMeta(jsonObject));

            if (isInExpectedHttpStatuses()) {
                resultBuilder.setValue(buildFunction.apply(jsonObject));
            }

            return resultBuilder.build();
        }

        /**
         * To list.
         *
         * @param jsonEntry     the json entry
         * @param buildFunction the build function
         * @return the http result
         */
        public HttpResult<List<T>> toList(final String jsonEntry, SDKFunction<JSONObject, T> buildFunction) {

            final HttpResult.Builder<List<T>> resultBuilder = new HttpResult.Builder<List<T>>()
                    .setETagValue(httpResponse.header(ETag.HEADER_NAME_RESPONSE))
                    .setHttpStatus(actualHttpStatus)
                    .setReasonPhrase(httpResponse.message());


            if (!jsonObjectOptional.isPresent()) {
                return resultBuilder.build();
            }

            final JSONObject jsonObject = jsonObjectOptional.get();

            resultBuilder.setSearchMeta(JSONUtils.transformOrReturnNull(jsonObject, SearchMeta.JSON_ENTRY, new SDKFunction<JSONObject, SearchMeta>() {
                @Override
                public SearchMeta apply(JSONObject object) {
                    try {
                        return SearchMeta.from(jsonObject.getJSONObject(SearchMeta.JSON_ENTRY));
                    } catch (JSONException e) {
                        throw new IllegalStateException(e);
                    }
                }
            }))
                    .setETagValue(httpResponse.header(ETag.HEADER_NAME_RESPONSE))
                    .setHttpStatus(actualHttpStatus)
                    .setSearchMeta(parseSearchMeta(jsonObject));
            try {

                if (isInExpectedHttpStatuses()) {
                    final JSONArray jsonArray = jsonObject.getJSONArray(jsonEntry);
                    final List<T> itemList = new ArrayList<T>();

                    for (int i = 0, size = jsonArray.length(); i < size; i++) {
                        itemList.add(buildFunction.apply(jsonArray.getJSONObject(i)));
                    }

                    resultBuilder.setValue(Collections.unmodifiableList(itemList));
                }

                return resultBuilder.build();
            } catch (Exception e) {
                throw new IllegalStateException(e);
            }
        }
    }

    /**
     * The type Byte array response.
     *
     * @param <T> the type parameter
     */
    public final class ByteArrayResponse<T> {

        /**
         * The Content.
         */
        final byte[] content;

        private ByteArrayResponse(byte[] content) {
            this.content = content;
        }

        /**
         * To http result.
         *
         * @param buildFunction the build function
         * @return the http result
         */
        public HttpResult<T> toHttpResult(final SDKFunction<byte[], T> buildFunction) {
            final HttpResult.Builder<T> builder = new HttpResult.Builder<T>()
                    .setHttpStatus(actualHttpStatus);
            if (isInExpectedHttpStatuses()) {
                builder.setValue(buildFunction.apply(content));
            }
            builder.setETagValue(httpResponse.header(ETag.HEADER_NAME_RESPONSE));

            return builder.build();
        }
    }

    /**
     * Parse search meta.
     *
     * @param jsonObject the json object
     * @return the search meta
     */
    static SearchMeta parseSearchMeta(final JSONObject jsonObject) {
        try {
            return JSONUtils.hasJSONKey(jsonObject, SearchMeta.JSON_ENTRY) ?
                    JSONUtils.transformOrThrow(jsonObject.getJSONObject(SearchMeta.JSON_ENTRY), new SDKThrowableFunction<JSONObject, SearchMeta, Exception>() {
                        @Override
                        public SearchMeta apply(JSONObject object) throws Exception {
                            return SearchMeta.from(object);
                        }
                    }) : null;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }
}
