package com.kontakt.sdk.android.http;

import android.text.TextUtils;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.model.BeaconId;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.DeviceCredentials;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.EddystoneFutureUID;
import com.kontakt.sdk.android.common.model.EddystoneUID;
import com.kontakt.sdk.android.common.model.IBeaconFutureId;
import com.kontakt.sdk.android.common.model.ICloudConfig;
import com.kontakt.sdk.android.common.model.IConfig;
import com.kontakt.sdk.android.common.model.ICredentials;
import com.kontakt.sdk.android.common.model.IDevice;
import com.kontakt.sdk.android.common.model.SecureSingleConfig;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.data.DeviceData;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.DevicesApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.kontakt.sdk.android.http.interfaces.UpdateApiCallback;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.HttpUtils.toUrlParameter;
import static com.kontakt.sdk.android.common.util.HttpUtils.toUrlParameterList;
import static com.kontakt.sdk.android.http.ApiConstants.Configs.CONFIG_JSON_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Devices.BEACON_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Devices.COMPANY_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Devices.DEVICES_JSON_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Devices.DEVICE_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Devices.MANAGER_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Devices.UNIQUE_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Devices.VENUE_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.BEACON_UPDATE;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICES_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_ASSIGN_TO_MANAGER;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_ASSIGN_TO_VENUE;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_CREDENTIALS_BATCH_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_CREDENTIAL_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_MOVE;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_UNASSIGNED_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Devices.DEVICE_UPDATE;

/**
 * {@link DevicesApiAccessor} implementation.
 */
final class DevicesApiAccessorImpl extends AbstractApiAccessor implements DevicesApiAccessor {


    /**
     * Instantiates a new BeaconsApiClientDelegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    DevicesApiAccessorImpl(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
    }

    @Override
    public HttpResult<List<IDevice>> getDevicesByProximity(UUID proximityUUID, int major, int minor, SDKOptional<ETag> eTag) throws ClientException {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Constants.Devices.PROXIMITY, proximityUUID.toString())
                .addParameter(Constants.Devices.MAJOR, String.valueOf(major))
                .addParameter(Constants.Devices.MINOR, String.valueOf(minor))
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return getAndTransform(DEVICES_GET,
                requestDescription,
                new SDKFunction<JSONObject, List<IDevice>>() {
                    @Override
                    public List<IDevice> apply(JSONObject object) {
                        return Device.fromList(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IDevice>> getDevicesByProximity(UUID proximityUUID, int major, int minor) throws ClientException {
        return getDevicesByProximity(proximityUUID, major, minor, SDKOptional.<ETag>absent());
    }

    @Override
    public void getDevicesByProximity(UUID proximityUUID, int major, int minor, SDKOptional<ETag> eTag, ResultApiCallback<List<IDevice>> apiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Constants.Devices.PROXIMITY, proximityUUID.toString())
                .addParameter(Constants.Devices.MAJOR, String.valueOf(major))
                .addParameter(Constants.Devices.MINOR, String.valueOf(minor))
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        getAsync(DEVICES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, List<IDevice>>() {
                    @Override
                    public List<IDevice> apply(JSONObject object) {
                        return Device.fromList(object);
                    }
                });
    }

    @Override
    public void getDevicesByProximity(UUID proximityUUID, int major, int minor, ResultApiCallback<List<IDevice>> apiCallback) {
        getDevicesByProximity(proximityUUID, major, minor, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<IDevice> getDeviceByNamespaceAndInstanceId(String namespace, String instanceId) throws ClientException {
        return getDeviceByNamespaceAndInstanceId(namespace, instanceId, SDKOptional.<ETag>absent());
    }

    @Override
    public HttpResult<IDevice> getDeviceByNamespaceAndInstanceId(String namespace, String instanceId, SDKOptional<ETag> eTag) throws ClientException {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Constants.Eddystone.INSTANCE_ID, instanceId)
                .addParameter(Constants.Eddystone.NAMESPACE, namespace)
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return getAndTransform(DEVICES_GET,
                requestDescription,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public void getDeviceByNamespaceAndInstanceId(String namespace, String instanceId, ResultApiCallback<IDevice> apiCallback) {
        getDeviceByNamespaceAndInstanceId(namespace, instanceId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getDeviceByNamespaceAndInstanceId(String namespace, String instanceId, SDKOptional<ETag> eTag, ResultApiCallback<IDevice> apiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Constants.Eddystone.INSTANCE_ID, instanceId)
                .addParameter(Constants.Eddystone.NAMESPACE, namespace)
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        getAsync(DEVICES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IDevice> getDevice(String beaconUniqueId, SDKOptional<ETag> eTag) throws ClientException {

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return getAndTransform(String.format(DEVICE_GET, beaconUniqueId),
                requestDescription,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IDevice> getDevice(String deviceUniqueId) throws ClientException {
        return getDevice(deviceUniqueId, SDKOptional.<ETag>absent());
    }


    @Override
    public void getDevice(String deviceUniqueId, SDKOptional<ETag> etag, ResultApiCallback<IDevice> apiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();

        String uri = String.format(DEVICE_GET, deviceUniqueId);

        getAsync(uri,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public void getDevice(String deviceUniqueId, ResultApiCallback<IDevice> apiCallback) {
        getDevice(deviceUniqueId, SDKOptional.<ETag>absent(), apiCallback);
    }


    @Override
    public HttpResult<List<IDevice>> listDevicesForManagers(Set<UUID> managerIds, SDKOptional<ETag> eTag) throws ClientException {

        RequestDescription description = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameters(HttpUtils.toUrlParameterList(ApiConstants.Devices.MANAGER_ID_PARAMETER, managerIds))
                .build();
        return getAndTransformToList(DEVICES_GET,
                description,
                DEVICES_JSON_PARAMETER,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IDevice>> listDevicesForManagers(Set<UUID> managerIds) throws ClientException {
        return listDevicesForManagers(managerIds, SDKOptional.<ETag>absent());
    }


    @Override
    public void listDevicesForManagers(Set<UUID> managerIds, SDKOptional<ETag> eTag, ResultApiCallback<List<IDevice>> apiCallback) {
        RequestDescription description = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameters(HttpUtils.toUrlParameterList(ApiConstants.Devices.MANAGER_ID_PARAMETER, managerIds))
                .build();

        transformToListAsynchronously(DEVICES_GET,
                description,
                HttpUtils.SC_OK,
                DEVICES_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public void listDevicesForManagers(Set<UUID> managerIds, ResultApiCallback<List<IDevice>> apiCallback) {
        listDevicesForManagers(managerIds, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<List<IDevice>> listDevices() throws ClientException {
        return listDevices(DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public HttpResult<List<IDevice>> listDevices(RequestDescription requestDescription) throws ClientException {
        return getAndTransformToList(DEVICES_GET,
                requestDescription,
                DEVICES_JSON_PARAMETER,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public void listDevices(ResultApiCallback<List<IDevice>> apiCallback) {
        listDevices(DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public void listDevices(RequestDescription requestDescription, ResultApiCallback<List<IDevice>> apiCallback) {
        transformToListAsynchronously(DEVICES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                DEVICES_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }


    @Override
    public HttpResult<ICredentials> getDeviceCredentials(final String deviceUniqueId, final SDKOptional<ETag> eTagOptional) throws ClientException {
        String endpoint = String.format(DEVICE_CREDENTIAL_GET, deviceUniqueId);

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTagOptional.isPresent() ? eTagOptional.get() : null)
                .build();

        return getAndTransform(endpoint, requestDescription, new SDKFunction<JSONObject, ICredentials>() {
            @Override
            public ICredentials apply(JSONObject object) {
                try {
                    return new DeviceCredentials.Builder()
                            .setDeviceUniqueId(deviceUniqueId)
                            .setPassword(object.getString(Constants.Devices.PASSWORD))
                            .setMasterPassword(object.getString(Constants.Devices.MASTER_PASSWORD))
                            .build();
                } catch (JSONException e) {
                    throw new IllegalStateException(e);
                }
            }
        });
    }

    @Override
    public HttpResult<List<ICredentials>> listDevicesCredentials(Collection<String> uniqueIds, SDKOptional<ETag> eTag) throws ClientException {
        String endpoint = DEVICE_CREDENTIALS_BATCH_GET;

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameters(HttpUtils.toUrlParameterList(Constants.UNIQUE_ID, uniqueIds))
                .setOffset(uniqueIds.size())
                .build();

        return getAndTransformToList(endpoint,
                requestDescription,
                Constants.Devices.CREDENTIALS,
                new SDKFunction<JSONObject, ICredentials>() {
                    @Override
                    public ICredentials apply(JSONObject object) {
                        return new DeviceCredentials.Builder()
                                .setDeviceUniqueId(JSONUtils.getString(object, Constants.UNIQUE_ID, null))
                                .setPassword(JSONUtils.getString(object, Constants.Devices.PASSWORD, null))
                                .setMasterPassword(JSONUtils.getString(object, Constants.Devices.MASTER_PASSWORD, null))
                                .build();
                    }
                });
    }

    @Override
    public void listDevicesCredentials(Collection<String> uniqueIds, SDKOptional<ETag> eTag, ResultApiCallback<List<ICredentials>> resultApiCallback) {
        String endpoint = DEVICE_CREDENTIALS_BATCH_GET;

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameters(HttpUtils.toUrlParameterList(Constants.UNIQUE_ID, uniqueIds))
                .setOffset(uniqueIds.size())
                .build();

        String jsonEntry = Constants.Devices.CREDENTIALS;

        final int expectedSuccessHttpCode = HttpUtils.SC_OK;

        transformToListAsynchronously(endpoint,
                requestDescription,
                expectedSuccessHttpCode,
                jsonEntry,
                resultApiCallback,
                new SDKFunction<JSONObject, ICredentials>() {
                    @Override
                    public ICredentials apply(JSONObject object) {
                        return new DeviceCredentials.Builder()
                                .setDeviceUniqueId(JSONUtils.getString(object, Constants.UNIQUE_ID, null))
                                .setPassword(JSONUtils.getString(object, Constants.Devices.PASSWORD, null))
                                .setMasterPassword(JSONUtils.getString(object, Constants.Devices.MASTER_PASSWORD, null))
                                .build();
                    }
                }
        );
    }

    @Override
    public void getDeviceCredentials(final String deviceUniqueId,
                                     final SDKOptional<ETag> etag,
                                     final ResultApiCallback<ICredentials> apiCallback) {
        String endpoint = String.format(DEVICE_CREDENTIAL_GET, deviceUniqueId);

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();

        getAsync(endpoint,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, ICredentials>() {
                    @Override
                    public ICredentials apply(JSONObject object) {
                        try {
                            return new DeviceCredentials.Builder()
                                    .setDeviceUniqueId(deviceUniqueId)
                                    .setPassword(object.getString(Constants.Devices.PASSWORD))
                                    .setMasterPassword(object.getString(Constants.Devices.MASTER_PASSWORD))
                                    .build();
                        } catch (JSONException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public int assignDevicesToManager(UUID managerId, Set<UUID> beaconIdSet) throws ClientException {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(MANAGER_ID_PARAMETER, managerId.toString())
                .addParameters(toUrlParameterList(BEACON_ID_PARAMETER, beaconIdSet))
                .build();

        return postAndReturnHttpStatus(DEVICE_ASSIGN_TO_MANAGER, requestDescription);
    }

    @Override
    public void assignDevicesToManager(UUID managerId, Set<UUID> beaconIdSet, UpdateApiCallback apiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(MANAGER_ID_PARAMETER, managerId.toString())
                .addParameters(toUrlParameterList(BEACON_ID_PARAMETER, beaconIdSet))
                .build();

        postAsyncAndReturnHttpStatus(DEVICE_ASSIGN_TO_MANAGER, requestDescription, apiCallback);
    }

    @Override
    public int applyConfig(IConfig config) throws ClientException {
        return updateDevice(createConfigData(config));
    }

    @Override
    public void applyConfig(IConfig config, UpdateApiCallback apiCallback) {
        updateDevice(createConfigData(config), apiCallback);
    }

    @Override
    public int applyCloudConfig(ICloudConfig config) throws ClientException {
        return updateDevice(buildIBeaconDeviceData(config));
    }

    @Override
    public void applyConfig(ICloudConfig config, UpdateApiCallback apiCallback) {
        updateDevice(buildIBeaconDeviceData(config), apiCallback);
    }


    @Override
    public HttpResult<List<SecureSingleConfig>> applySecureConfig(Collection<SecureSingleConfig> secureConfigApplies) throws ClientException {
        String endpoint = ApiMethods.Devices.DEVICE_UPDATE_SECURE_CONFIG;

        RequestDescription requestDescription = ObjectRequestDescriptionHelper.prepareParameters(secureConfigApplies)
                .setOffset(secureConfigApplies.size())
                .build();

        int[] statuses = {HttpUtils.SC_OK, HttpUtils.SC_NO_CONTENT};

        return postAndTransformToList(endpoint,
                requestDescription,
                statuses,
                CONFIG_JSON_PARAMETER,
                new SDKFunction<JSONObject, SecureSingleConfig>() {
                    @Override
                    public SecureSingleConfig apply(JSONObject object) {
                        return SecureSingleConfig.fromJson(object);
                    }
                });

    }

    @Override
    public void applySecureConfig(Collection<SecureSingleConfig> secureConfigApplies, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        String endpoint = ApiMethods.Devices.DEVICE_UPDATE_SECURE_CONFIG;

        RequestDescription requestDescription = ObjectRequestDescriptionHelper.prepareParameters(secureConfigApplies)
                .setOffset(secureConfigApplies.size())
                .build();

        int[] statuses = {HttpUtils.SC_OK, HttpUtils.SC_NO_CONTENT};

        postAsyncAndBuildFromJSONObject(endpoint,
                requestDescription,
                statuses,
                apiCallback,
                new SDKFunction<JSONObject, List<SecureSingleConfig>>() {
                    @Override
                    public List<SecureSingleConfig> apply(JSONObject object) {
                        return SecureSingleConfig.fromListJson(object);
                    }
                });

    }

    @Override
    public HttpResult<List<IDevice>> listUnassignedDevicesForManager(UUID managerId, RequestDescription requestDescription) throws ClientException {

        return getAndTransformToList(String.format(DEVICE_UNASSIGNED_GET, managerId.toString()),
                requestDescription,
                DEVICES_JSON_PARAMETER,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IDevice>> listUnassignedDevicesForManager(UUID managerId) throws ClientException {
        return listUnassignedDevicesForManager(managerId, DEFAULT_REQUEST_DESCRIPTION);
    }


    @Override
    public void listUnassignedDevicesForManager(UUID managerId, RequestDescription requestDescription, ResultApiCallback<List<IDevice>> apiCallback) {
        final String uri = String.format(DEVICE_UNASSIGNED_GET, managerId.toString());
        transformToListAsynchronously(uri,
                requestDescription,
                HttpUtils.SC_OK,
                DEVICES_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, IDevice>() {
                    @Override
                    public IDevice apply(JSONObject object) {
                        return Device.from(object);
                    }
                });
    }

    @Override
    public void listUnassignedDevicesForManager(UUID managerId, ResultApiCallback<List<IDevice>> apiCallback) {
        listUnassignedDevicesForManager(managerId, DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public void resolveIBeacon(Collection<BeaconId> beaconIds, SDKOptional<ETag> eTag, ResultApiCallback<List<IBeaconFutureId>> apiCallback) {
        String parameters = TextUtils.join(",", beaconIds);

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameter("bid", parameters)
                .setOffset(beaconIds.size())
                .build();

        transformToListAsynchronously(DEVICES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                DEVICES_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, IBeaconFutureId>() {
                    @Override
                    public IBeaconFutureId apply(JSONObject object) {
                        return IBeaconFutureId.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IBeaconFutureId>> resolveIBeacon(Collection<BeaconId> beaconIds, SDKOptional<ETag> eTagSDKOptional) throws ClientException {
        String parameters = TextUtils.join(",", beaconIds);
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTagSDKOptional.isPresent() ? eTagSDKOptional.get() : null)
                .addParameter("bid", parameters)
                .setOffset(beaconIds.size())
                .build();


        return getAndTransformToList(DEVICES_GET,
                requestDescription,
                DEVICES_JSON_PARAMETER,
                new SDKFunction<JSONObject, IBeaconFutureId>() {
                    @Override
                    public IBeaconFutureId apply(JSONObject object) {
                        return IBeaconFutureId.from(object);
                    }
                });
    }

    @Override
    public void resolveEddystone(Collection<EddystoneUID> eddystoneUids, SDKOptional<ETag> eTag, ResultApiCallback<List<EddystoneFutureUID>> apiCallback) {
        String parameters = TextUtils.join(",", eddystoneUids);

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameter("euid", parameters)
                .setOffset(eddystoneUids.size())
                .build();

        transformToListAsynchronously(DEVICES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                DEVICES_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, EddystoneFutureUID>() {
                    @Override
                    public EddystoneFutureUID apply(JSONObject object) {
                        return EddystoneFutureUID.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<EddystoneFutureUID>> resolveEddystone(Collection<EddystoneUID> eddystoneUids, SDKOptional<ETag> eTagSDKOptional) throws ClientException {
        String parameters = TextUtils.join(",", eddystoneUids);
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTagSDKOptional.isPresent() ? eTagSDKOptional.get() : null)
                .addParameter("euid", parameters)
                .setOffset(eddystoneUids.size())
                .build();


        return getAndTransformToList(DEVICES_GET,
                requestDescription,
                DEVICES_JSON_PARAMETER,
                new SDKFunction<JSONObject, EddystoneFutureUID>() {
                    @Override
                    public EddystoneFutureUID apply(JSONObject object) {
                        return EddystoneFutureUID.from(object);
                    }
                });
    }

    @Override
    public int assignDevicesToVenue(UUID venueId, Set<UUID> deviceIdSet) throws ClientException {
        final List<Map.Entry<String, String>> parameterList = HttpUtils.toUrlParameterList(DEVICE_ID_PARAMETER, deviceIdSet);
        parameterList.add(toUrlParameter(VENUE_ID_PARAMETER, venueId.toString()));

        return postAndReturnHttpStatus(DEVICE_ASSIGN_TO_MANAGER, RequestDescription.start()
                .addParameters(parameterList)
                .build());
    }

    @Override
    public void assignDevicesToVenue(UUID venueId, Set<UUID> deviceIdSet, UpdateApiCallback apiUpdateCallback) {
        final List<Map.Entry<String, String>> parameterList = HttpUtils.toUrlParameterList(DEVICE_ID_PARAMETER, deviceIdSet);
        parameterList.add(toUrlParameter(VENUE_ID_PARAMETER, venueId.toString()));

        postAsyncAndReturnHttpStatus(DEVICE_ASSIGN_TO_VENUE,
                RequestDescription.start().addParameters(parameterList).build(),
                apiUpdateCallback);
    }

    @Override
    public int updateDevice(final DeviceData deviceData) throws ClientException {
        return postAndReturnHttpStatus(DEVICE_UPDATE, RequestDescription.start()
                .addParameters(deviceData.getParameters())
                .build());
    }

    @Override
    public void updateDevice(DeviceData deviceData, UpdateApiCallback callback) {
        postAsyncAndReturnHttpStatus(DEVICE_UPDATE,
                RequestDescription.start().addParameters(deviceData.getParameters()).build(),
                callback);
    }


    @Override
    public int moveDevicesToManager(Set<String> beaconUniqueIds,
                                    UUID receiverId,
                                    UUID receiverCompanyId) throws ClientException {
        List<Map.Entry<String, String>> parameterList = HttpUtils.toUrlParameterList(UNIQUE_ID_PARAMETER, beaconUniqueIds);
        parameterList.add(HttpUtils.toUrlParameter(MANAGER_ID_PARAMETER, receiverId.toString()));
        parameterList.add(HttpUtils.toUrlParameter(COMPANY_ID_PARAMETER, receiverCompanyId.toString()));

        return postAndReturnHttpStatus(DEVICE_MOVE, RequestDescription.start()
                .addParameters(parameterList)
                .build());
    }

    @Override
    public void moveDevicesToManager(Set<String> deviceUniqueIds, UUID receiverID, UUID receiverCompanyId, UpdateApiCallback apiUpdateCallback) {
        List<Map.Entry<String, String>> parameterList = toUrlParameterList(UNIQUE_ID_PARAMETER, deviceUniqueIds);
        parameterList.add(toUrlParameter(MANAGER_ID_PARAMETER, receiverID.toString()));
        parameterList.add(toUrlParameter(COMPANY_ID_PARAMETER, receiverCompanyId.toString()));

        postAsyncAndReturnHttpStatus(DEVICE_MOVE,
                RequestDescription.start()
                        .addParameters(parameterList)
                        .build(),
                apiUpdateCallback);
    }

    @Override
    public int updateDevicePassword(final String beaconUniqueId, final String password) throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Constants.UNIQUE_ID, beaconUniqueId)
                .addParameter(Constants.Devices.PASSWORD, password)
                .build();

        return postAndReturnHttpStatus(BEACON_UPDATE, requestDescription);
    }

    @Override
    public void updateDevicePassword(String beaconUniqueId, String password, UpdateApiCallback callback) {

        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Constants.UNIQUE_ID, beaconUniqueId)
                .addParameter(Constants.Devices.PASSWORD, password)
                .build();

        postAsyncAndReturnHttpStatus(BEACON_UPDATE,
                requestDescription,
                callback);
    }

    private DeviceData buildIBeaconDeviceData(final ICloudConfig config) {
        return DeviceData.update(DeviceType.CLOUD_BEACON, config.getDeviceUniqueId())
                .withWifiScanInterval(config.getWifiScanInterval())
                .withBleScanDuration(config.getBleScanDuration())
                .withInterval(config.getInterval())
                .withWorkingMode(config.getWorkingMode())
                .withDefaultSSIDAuth(config.getDefaultSSIDAuth())
                .withMinor(config.getMinor())
                .withTxPower(config.getTxPower())
                .withPassword(config.getPassword())
                .withDataSendInterval(config.getDataSendInterval())
                .withProximityUUID(config.getProximityUUID())
                .withDefaultSSIDCrypt(config.getDefaultSSIDCrypt())
                .withBleScanInterval(config.getBleScanInterval())
                .withDefaultSSIDName(config.getDefaultSSIDName())
                .withName(config.getName())
                .withMajor(config.getMajor())
                .build();
    }


    private DeviceData createConfigData(IConfig config) {
        DeviceData.Builder builder = DeviceData.update(DeviceType.BEACON, config.getDeviceUniqueId())
                .withTxPower(config.getTxPower());

        if (config.getMajor() > 0 && config.getMajor() < 65535) {
            builder.withMajor(config.getMajor());
        }

        if (config.getMinor() > 0 && config.getMinor() < 65535) {
            builder.withMinor(config.getMinor());
        }

        if (config.getUrl() != null) {
            builder.withUrl(EddystoneUtils.toHexString(EddystoneUtils.serializeUrl(config.getUrl())));
        }

        if (config.getInterval() > 0) {
            builder.withInterval(config.getInterval());
        }

        if (config.getNamespace() != null) {
            builder.withNamespace(config.getNamespace());
        }

        if (config.getProximityUUID() != null) {
            builder.withProximityUUID(config.getProximityUUID());
        }

        if (config.getInstanceId() != null) {
            builder.withInstanceId(config.getInstanceId());
        }

        return builder.build();

    }
}
