package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.model.INamespace;
import com.kontakt.sdk.android.common.model.IProximityUUID;
import com.kontakt.sdk.android.common.model.EventPacket;
import com.kontakt.sdk.android.common.model.Namespace;
import com.kontakt.sdk.android.common.model.ProximityUUID;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.CommonApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;

import org.json.JSONObject;

import java.util.Collections;
import java.util.List;
import java.util.UUID;

import static com.kontakt.sdk.android.http.ApiMethods.Monitoring.EVENT_COLLECT;
import static com.kontakt.sdk.android.http.ApiMethods.Namespaces.NAMESPACES_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Proximities.PROXIMITIES_GET;

final class CommonApiAccessorImpl extends AbstractApiAccessor implements CommonApiAccessor {

    /**
     * Instantiates a new Abstract api client delegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    CommonApiAccessorImpl(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
    }

    @Override
    public HttpResult<List<IProximityUUID>> listProximities() throws ClientException {
        return listProximities(SDKOptional.<ETag>absent());
    }

    @Override
    public HttpResult<List<IProximityUUID>> listProximities(final SDKOptional<ETag> eTag) throws ClientException {
        return resolveProximities(Collections.EMPTY_LIST, eTag);

    }

    @Override
    public void listProximities(ResultApiCallback<List<IProximityUUID>> apiCallback) {
        listProximities(SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void listProximities(SDKOptional<ETag> etag, ResultApiCallback<List<IProximityUUID>> apiCallback) {
        resolveProximities(Collections.EMPTY_LIST, etag, apiCallback);
    }

    @Override
    public HttpResult<List<IProximityUUID>> resolveProximities(List<UUID> proximities) throws ClientException {
        return resolveProximities(proximities, SDKOptional.<ETag>absent());
    }

    @Override
    public HttpResult<List<IProximityUUID>> resolveProximities(List<UUID> proximities, SDKOptional<ETag> etag) throws ClientException {
        try {
            SDKPreconditions.checkNotNull(proximities, "Arguments array is null.");
            RequestDescription.Descriptor builder = RequestDescription.start()
                    .setETag(etag.isPresent() ? etag.get() : null);

            if (!proximities.isEmpty()) {
                builder.addParameters(HttpUtils.toUrlParameterList(ApiConstants.Proximities.PROXIMITY_PARAMETER, proximities));
            }

            RequestDescription requestDescription = builder.build();

            return getAndTransform(PROXIMITIES_GET,
                    requestDescription,
                    new SDKFunction<JSONObject, List<IProximityUUID>>() {
                        @Override
                        public List<IProximityUUID> apply(JSONObject object) {
                            return ProximityUUID.fromList(object);
                        }
                    });

        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    @Override
    public void resolveProximities(List<UUID> proximities, ResultApiCallback<List<IProximityUUID>> apiCallback) {
        resolveProximities(proximities, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void resolveProximities(List<UUID> proximities, SDKOptional<ETag> etag, ResultApiCallback<List<IProximityUUID>> apiCallback) {
        SDKPreconditions.checkNotNull(proximities, "Arguments array is null.");
        RequestDescription.Descriptor builder = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null);

        if (!proximities.isEmpty()) {
            builder.addParameters(HttpUtils.toUrlParameterList(ApiConstants.Proximities.PROXIMITY_PARAMETER, proximities));
        }

        final RequestDescription requestDescription = builder.build();

        getAsync(PROXIMITIES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, List<IProximityUUID>>() {
                    @Override
                    public List<IProximityUUID> apply(JSONObject object) {
                        return ProximityUUID.fromList(object);
                    }
                });
    }

    @Override
    public HttpResult<List<INamespace>> listNamespaces() throws ClientException {
        return listNamespaces(SDKOptional.<ETag>absent());
    }

    @Override
    public HttpResult<List<INamespace>> listNamespaces(SDKOptional<ETag> etag) throws ClientException {
        return resolveNamespaces(Collections.EMPTY_LIST, etag);
    }

    @Override
    public void listNamespaces(ResultApiCallback<List<INamespace>> apiCallback) {
        listNamespaces(SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void listNamespaces(SDKOptional<ETag> etag, ResultApiCallback<List<INamespace>> apiCallback) {
        resolveNamespaces(Collections.EMPTY_LIST, etag, apiCallback);
    }

    @Override
    public HttpResult<List<INamespace>> resolveNamespaces(List<String> namespaces) throws ClientException {
        return resolveNamespaces(namespaces, SDKOptional.<ETag>absent());
    }

    @Override
    public HttpResult<List<INamespace>> resolveNamespaces(List<String> namespaces, SDKOptional<ETag> etag) throws ClientException {
        try {
            SDKPreconditions.checkNotNull(namespaces, "Arguments array is null.");
            RequestDescription.Descriptor builder = RequestDescription.start()
                    .setETag(etag.isPresent() ? etag.get() : null);

            if (!namespaces.isEmpty()) {
                builder.addParameters(HttpUtils.toUrlParameterList(ApiConstants.Namespaces.NAMESPACE_PARAMETER, namespaces));
            }

            RequestDescription requestDescription = builder.build();

            return getAndTransform(NAMESPACES_GET,
                    requestDescription,
                    new SDKFunction<JSONObject, List<INamespace>>() {
                        @Override
                        public List<INamespace> apply(JSONObject object) {
                            return Namespace.fromList(object);
                        }
                    });

        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    @Override
    public void resolveNamespaces(List<String> namespaces, ResultApiCallback<List<INamespace>> apiCallback) {
        resolveNamespaces(namespaces, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void resolveNamespaces(List<String> namespaces, SDKOptional<ETag> etag, ResultApiCallback<List<INamespace>> apiCallback) {
        SDKPreconditions.checkNotNull(namespaces, "Arguments array is null.");
        RequestDescription.Descriptor builder = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null);

        if (!namespaces.isEmpty()) {
            builder.addParameters(HttpUtils.toUrlParameterList(ApiConstants.Namespaces.NAMESPACE_PARAMETER, namespaces));
        }

        final RequestDescription requestDescription = builder.build();

        getAsync(NAMESPACES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, List<INamespace>>() {
                    @Override
                    public List<INamespace> apply(JSONObject object) {
                        return Namespace.fromList(object);
                    }
                });
    }

    @Override
    public int sendEvents(EventPacket eventPacket) throws ClientException {
        RequestDescription requestDescription = RequestDescription.start().build();
        return postAndReturnHttpStatus(EVENT_COLLECT, requestDescription, eventPacket, new SDKFunction<EventPacket, JSONObject>() {
            @Override
            public JSONObject apply(EventPacket packet) {
                return EventPacket.toJSONObject(packet);
            }
        });
    }

}
