package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.model.BeaconId;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.EddystoneFutureUID;
import com.kontakt.sdk.android.common.model.EddystoneUID;
import com.kontakt.sdk.android.common.model.EventPacket;
import com.kontakt.sdk.android.common.model.IAction;
import com.kontakt.sdk.android.common.model.IBeaconFutureId;
import com.kontakt.sdk.android.common.model.IBrowserAction;
import com.kontakt.sdk.android.common.model.ICloudConfig;
import com.kontakt.sdk.android.common.model.IConfig;
import com.kontakt.sdk.android.common.model.IContentAction;
import com.kontakt.sdk.android.common.model.ICredentials;
import com.kontakt.sdk.android.common.model.IDevice;
import com.kontakt.sdk.android.common.model.IFirmware;
import com.kontakt.sdk.android.common.model.IManager;
import com.kontakt.sdk.android.common.model.INamespace;
import com.kontakt.sdk.android.common.model.IPreset;
import com.kontakt.sdk.android.common.model.IProximityUUID;
import com.kontakt.sdk.android.common.model.IVenue;
import com.kontakt.sdk.android.common.model.SecureCommandResponse;
import com.kontakt.sdk.android.common.model.SecureCommandType;
import com.kontakt.sdk.android.common.model.SecureSingleConfig;
import com.kontakt.sdk.android.common.util.IBeaconPropertyValidator;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import com.kontakt.sdk.android.http.data.ActionData;
import com.kontakt.sdk.android.http.data.ConfigData;
import com.kontakt.sdk.android.http.data.DeviceData;
import com.kontakt.sdk.android.http.data.ManagerData;
import com.kontakt.sdk.android.http.data.VenueData;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.ActionsApiAccessor;
import com.kontakt.sdk.android.http.interfaces.CommandApiAccessor;
import com.kontakt.sdk.android.http.interfaces.CommonApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ConfigurationApiAccessor;
import com.kontakt.sdk.android.http.interfaces.DevicesApiAccessor;
import com.kontakt.sdk.android.http.interfaces.FirmwareApiAccessor;
import com.kontakt.sdk.android.http.interfaces.IKontaktApiClient;
import com.kontakt.sdk.android.http.interfaces.ManagersApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.kontakt.sdk.android.http.interfaces.UpdateApiCallback;
import com.kontakt.sdk.android.http.interfaces.VenuesApiAccessor;
import java.io.File;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 * AbstractApiClient provides abstraction for both
 * {@link KontaktApiClient} and
 */
abstract class AbstractKontaktApiClient implements IKontaktApiClient {
    /**
     * Returns implementations of {@link DevicesApiAccessor}.
     *
     * @return the beacons api accessor
     */
    protected abstract DevicesApiAccessor devicesApi();

    /**
     * Returns implementation of {com.kontakt.sdk.http.interfaces.VenuesApiAccessor}.
     *
     * @return the venues api accessor
     */
    protected abstract VenuesApiAccessor venuesApi();

    /**
     * Returns implementations of {com.kontakt.sdk.http.interfaces.ManagersApiAccessor}.
     *
     * @return the managers api accessor
     */
    protected abstract ManagersApiAccessor managersApi();

    /**
     * returns implementation of {com.kontakt.sdk.http.interfaces.ConfigurationApiAccessor}.
     *
     * @return the configuration api accessor
     */
    protected abstract ConfigurationApiAccessor configurationApi();

    /**
     * Returns implementation of {com.kontakt.sdk.http.interfaces.FirmwareApiAccessor}.
     *
     * @return the firmware api accessor
     */
    protected abstract FirmwareApiAccessor firmwareApi();

    /**
     * Returns implementation of {com.kontakt.sdk.http.interfaces.ActionsApiAccessor}.
     *
     * @return the actions api accessor
     */
    protected abstract ActionsApiAccessor actionsApi();

    /**
     * Returns implementation of .
     *
     * @return the common api accessor
     */
    protected abstract CommonApiAccessor commonsApi();

    protected abstract CommandApiAccessor commandApi();

    /**
     * The constant ACCEPT_VERSION.
     */
    protected static final int ACCEPT_VERSION = 9;

    /**
     * The Api key.
     */
    protected final String apiKey;

    /**
     * The Api url.
     */
    protected final String apiUrl;

    /**
     * Instantiates a new Abstract REST client.
     *
     * @param apiKey the API key
     * @param apiUrl the API url
     */
    protected AbstractKontaktApiClient(String apiKey, String apiUrl) {
        SDKPreconditions.checkNotNullOrEmpty(apiKey, "Api Client: Empty Api-Key.");
        SDKPreconditions.checkNotNullOrEmpty(apiUrl, "Api Client: Empty Api Url.");

        this.apiKey = apiKey;
        this.apiUrl = apiUrl;
    }

    /**
     * Gets beacon credentials.
     *
     * @param beaconUniqueId the beacon unique id
     * @return the beacon credentials
     * @throws ClientException the client exception
     */
    public HttpResult<ICredentials> getDeviceCredentials(final String beaconUniqueId) throws ClientException {
        return getDeviceCredentials(beaconUniqueId, SDKOptional.<ETag>absent());
    }

    /**
     * Gets beacon credentials.
     *
     * @param deviceUniqueId the beacon unique id
     * @param eTag           the e tag
     * @return the beacon credentials
     * @throws ClientException the client exception
     */
    public HttpResult<ICredentials> getDeviceCredentials(final String deviceUniqueId, final SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(deviceUniqueId, "Beacon unique ID is null or empty.");
        RequestValidator.validateETag(eTag);

        return devicesApi().getDeviceCredentials(deviceUniqueId, eTag);
    }

    public void getDeviceCredentials(String deviceUniqueId, ResultApiCallback<ICredentials> apiCallback) {
        getDeviceCredentials(deviceUniqueId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getDeviceCredentials(String deviceUniqueId, SDKOptional<ETag> etag, ResultApiCallback<ICredentials> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(deviceUniqueId, "Beacon unique ID is null or empty.");
        RequestValidator.validateETag(etag);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null");

        devicesApi().getDeviceCredentials(deviceUniqueId, etag, apiCallback);
    }

    /**
     * Gets device.
     *
     * @param beaconUniqueId the beacon unique id
     * @return the beacon
     * @throws ClientException the client exception
     */
    public HttpResult<IDevice> getDevice(final String beaconUniqueId) throws ClientException {
        return getDevice(beaconUniqueId, SDKOptional.<ETag>absent());
    }

    /**
     * Gets device.
     *
     * @param beaconUniqueId the beacon unique id
     * @param eTag           the e tag
     * @return the device
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IDevice> getDevice(String beaconUniqueId, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueId, "Beacon unique id is null");
        RequestValidator.validateETag(eTag);

        return devicesApi().getDevice(beaconUniqueId, eTag);
    }


    @Override
    public void getDevice(String deviceUniqueId, ResultApiCallback<IDevice> apiCallback) {
        getDevice(deviceUniqueId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getDevice(String deviceUniqueId, SDKOptional<ETag> etag, ResultApiCallback<IDevice> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(deviceUniqueId, "Beacon unique id is null");
        RequestValidator.validateETag(etag);
        devicesApi().getDevice(deviceUniqueId, etag, apiCallback);
    }

    /**
     * Update beacon password.
     *
     * @param beaconUniqueId the beacon unique id
     * @param password       the password
     * @return 200 (OK) if update succeeds
     * @throws ClientException the client exception
     */
    public int updateDevicePassword(final String beaconUniqueId, final String password) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueId, "Beacon Unique Id is null.");
        IBeaconPropertyValidator.validateBeaconPassword(password, null);
        return devicesApi().updateDevicePassword(beaconUniqueId, password);
    }

    @Override
    public void updateDevicePassword(String beaconUniqueId, String password, UpdateApiCallback callback) {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueId, "Beacon Unique Id is null.");
        IBeaconPropertyValidator.validateBeaconPassword(password, null);
        devicesApi().updateDevicePassword(beaconUniqueId, password, callback);
    }

    /**
     * Gets beacons for managers.
     *
     * @param managerIds the manager ids
     * @return the beacons for managers
     * @throws ClientException the client exception
     */
    public HttpResult<List<IDevice>> listDevicesForManagers(Set<UUID> managerIds) throws ClientException {
        return listDevicesForManagers(managerIds, SDKOptional.<ETag>absent());
    }

    /**
     * Gets devices for managers.
     *
     * @param managerIds the manager ids
     * @param eTag       the e tag
     * @return the devices for managers
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<List<IDevice>> listDevicesForManagers(Set<UUID> managerIds, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(managerIds, "Managers array cannot be empty");
        RequestValidator.validateETag(eTag);

        return devicesApi().listDevicesForManagers(managerIds, eTag);
    }

    public void listDevicesForManagers(Set<UUID> managerIds, ResultApiCallback<List<IDevice>> apiCallback) {
        listDevicesForManagers(managerIds, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void listDevicesForManagers(Set<UUID> managerIds, SDKOptional<ETag> eTag, ResultApiCallback<List<IDevice>> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(managerIds, "Managers array cannot be empty");
        RequestValidator.validateETag(eTag);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null");

        devicesApi().listDevicesForManagers(managerIds, eTag, apiCallback);
    }

    /**
     * Assign devices to venue.
     *
     * @param venueId     the venue id
     * @param beaconIdSet the beacon id set
     * @return 200 (OK) if assignment succeeds
     * @throws ClientException the client exception
     */
    @Override
    public int assignDevicesToVenue(UUID venueId, Set<UUID> beaconIdSet) throws ClientException {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null");
        SDKPreconditions.checkNotNullOrEmpty(beaconIdSet, "Beacon Ids set is null");

        return devicesApi().assignDevicesToVenue(venueId, beaconIdSet);
    }

    @Override
    public void assignDevicesToVenue(UUID venueId, Set<UUID> deviceIdSet, UpdateApiCallback apiUpdateCallback) {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null");
        SDKPreconditions.checkNotNullOrEmpty(deviceIdSet, "Beacon Ids set is null");

        devicesApi().assignDevicesToVenue(venueId, deviceIdSet, apiUpdateCallback);
    }

    @Override
    public HttpResult<List<ICredentials>> listDevicesCredentials(Collection<String> uniqueIds, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(uniqueIds, "Unique ids are null or empty");
        SDKPreconditions.checkNotNull(eTag, "ETag is null or empty");

        return devicesApi().listDevicesCredentials(uniqueIds, eTag);
    }

    @Override
    public void listDevicesCredentials(Collection<String> uniqueIds, SDKOptional<ETag> eTag, ResultApiCallback<List<ICredentials>> resultApiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(uniqueIds, "Unique ids are null or empty");
        SDKPreconditions.checkNotNull(eTag, "ETag is null or empty");
        SDKPreconditions.checkNotNull(resultApiCallback);

        devicesApi().listDevicesCredentials(
                uniqueIds,
                eTag,
                resultApiCallback);
    }

    /**
     * Assign beacons to manager.
     *
     * @param managerId   the manager id
     * @param beaconIdSet the beacon id set
     * @return 200 (OK) if assignment succeeds
     * @throws ClientException the client exception
     */
    @Override
    public int assignDevicesToManager(UUID managerId, Set<UUID> beaconIdSet) throws ClientException {
        SDKPreconditions.checkNotNull(managerId, "Manager Id is null");
        SDKPreconditions.checkNotNullOrEmpty(beaconIdSet, "Beacon ids set is null");

        return devicesApi().assignDevicesToManager(managerId, beaconIdSet);
    }

    @Override
    public void assignDevicesToManager(UUID managerId, Set<UUID> beaconIdSet, UpdateApiCallback apiCallback) {
        SDKPreconditions.checkNotNull(managerId, "Manager Id is null");
        SDKPreconditions.checkNotNullOrEmpty(beaconIdSet, "Beacon ids set is null");

        devicesApi().assignDevicesToManager(managerId, beaconIdSet, apiCallback);
    }

    /**
     * Update device.
     *
     * @param deviceData the device data
     * @return 200 (OK) if update succeeds
     * @throws ClientException the client exception
     */
    @Override
    public int updateDevice(final DeviceData deviceData) throws ClientException {
        return devicesApi().updateDevice(deviceData);
    }

    @Override
    public void updateDevice(DeviceData deviceData, UpdateApiCallback callback) {
        devicesApi().updateDevice(deviceData, callback);
    }

    /**
     * Move beacons to manager.
     *
     * @param deviceUniqueIds   the beacon unique ids
     * @param receiverId        the receiver id
     * @param receiverCompanyId the receiver company id
     * @return the 200 (OK) if moving succeeds
     * @throws ClientException the client exception
     */
    @Override
    public int moveDevicesToManager(Set<String> deviceUniqueIds, UUID receiverId, UUID receiverCompanyId) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(deviceUniqueIds, "Beacon Unique Ids set is null.");
        SDKPreconditions.checkNotNull(receiverId, "Receiver Id is null.");
        SDKPreconditions.checkNotNull(receiverCompanyId, "Receiver's company Id is null.");

        return devicesApi().moveDevicesToManager(deviceUniqueIds, receiverId, receiverCompanyId);
    }

    @Override
    public void moveDevicesToManager(Set<String> deviceUniqueIds, UUID receiverId, UUID receiverCompanyId, UpdateApiCallback apiUpdateCallback) {
        SDKPreconditions.checkNotNullOrEmpty(deviceUniqueIds, "Beacon Unique Ids set is null.");
        SDKPreconditions.checkNotNull(receiverId, "Receiver Id is null.");
        SDKPreconditions.checkNotNull(receiverCompanyId, "Receiver's company Id is null.");

        devicesApi().moveDevicesToManager(deviceUniqueIds, receiverId, receiverCompanyId, apiUpdateCallback);
    }

    /**
     * Update venue.
     *
     * @param venueData the venue data
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int updateVenue(VenueData venueData) throws ClientException {

        return venuesApi().updateVenue(venueData);
    }

    @Override
    public void updateVenue(VenueData venueData, UpdateApiCallback apiCallback) {

        venuesApi().updateVenue(venueData, apiCallback);
    }

    /**
     * Create venue.
     *
     * @param venue the venue
     * @return the http result
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IVenue> createVenue(final VenueData venue) throws ClientException {
        SDKPreconditions.checkNotNull(venue, "Venue data is null.");

        return venuesApi().createVenue(venue);
    }

    @Override
    public void createVenue(VenueData venueData, ResultApiCallback<IVenue> apiCallback) {
        SDKPreconditions.checkNotNull(venueData, "Venue data is null.");

        venuesApi().createVenue(venueData, apiCallback);
    }

    /**
     * Gets venue.
     *
     * @param venueId the venue id
     * @return the venue
     * @throws ClientException the client exception
     */
    public HttpResult<IVenue> getVenue(final UUID venueId) throws ClientException {
        return getVenue(venueId, SDKOptional.<ETag>absent());
    }

    public void getVenue(UUID venueId, ResultApiCallback<IVenue> apiCallback) {
        getVenue(venueId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getVenue(UUID venueId, SDKOptional<ETag> etag, ResultApiCallback<IVenue> apiCallback) {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null.");

        venuesApi().getVenue(venueId, etag, apiCallback);
    }

    /**
     * Gets venue.
     *
     * @param venueId the venue id
     * @param eTag    the e tag
     * @return the venue
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IVenue> getVenue(UUID venueId, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkArgument(venueId != null, "Venue Id is null");
        RequestValidator.validateETag(eTag);

        return venuesApi().getVenue(venueId, eTag);
    }

    /**
     * Delete venue.
     *
     * @param venueId the venue id
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int deleteVenue(UUID venueId) throws ClientException {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null.");
        return venuesApi().deleteVenue(venueId);
    }

    @Override
    public void deleteVenue(UUID venueId, UpdateApiCallback apiCallback) {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null", venueId);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null");

        venuesApi().deleteVenue(venueId, apiCallback);
    }

    /**
     * Gets venue image.
     *
     * @param venueId the venue id
     * @return the venue image
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<FileData> getVenueImage(final UUID venueId) throws ClientException {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null.");
        return venuesApi().getVenueImage(venueId);
    }

    @Override
    public void getVenueImage(UUID venueId, ResultApiCallback<FileData> apiCallback) {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null.");
        venuesApi().getVenueImage(venueId, apiCallback);
    }

    /**
     * Gets subordinates for manager.
     *
     * @param managerId the manager id
     * @return the subordinates for manager
     * @throws ClientException the client exception
     */
    public HttpResult<List<IManager>> listSubordinatesForManager(UUID managerId) throws ClientException {
        return listSubordinatesForManager(managerId, AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION);
    }

    public void listSubordinatesForManager(UUID managerId,
                                           ResultApiCallback<List<IManager>> resultApiCallback) {
        listSubordinatesForManager(managerId, AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION, resultApiCallback);
    }

    @Override
    public void listSubordinatesForManager(UUID managerId,
                                           RequestDescription requestDescription,
                                           ResultApiCallback<List<IManager>> resultApiCallback) {
        SDKPreconditions.checkNotNull(managerId, "Manager id is null");
        SDKPreconditions.checkNotNull(requestDescription, "Request description is null.");
        managersApi().listSubordinatesForManager(managerId, requestDescription, resultApiCallback);
    }

    /**
     * Gets subordinates for manager.
     *
     * @param managerId the manager id
     * @return the subordinates for manager
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<List<IManager>> listSubordinatesForManager(UUID managerId, RequestDescription requestDescription) throws ClientException {
        SDKPreconditions.checkNotNull(managerId, "Manager Id is null.");
        SDKPreconditions.checkNotNull(requestDescription, "Request Description is null.");

        return managersApi().listSubordinatesForManager(managerId, requestDescription);
    }

    /**
     * Delete manager.
     *
     * @param managerId the manager id
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int deleteManager(UUID managerId) throws ClientException {
        SDKPreconditions.checkNotNull(managerId, "Manager Id is null.");
        return managersApi().deleteManager(managerId);
    }

    @Override
    public void deleteManager(UUID managerId, UpdateApiCallback apiCallback) {
        SDKPreconditions.checkNotNull(managerId, "Manager Id is null.");
        managersApi().deleteManager(managerId, apiCallback);
    }

    /**
     * Update manager.
     *
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int updateManager(ManagerData managerData) throws ClientException {
        SDKPreconditions.checkNotNull(managerData, "Manager data cannot be null");

        return managersApi().updateManager(managerData);
    }

    @Override
    public void updateManager(ManagerData managerData, UpdateApiCallback updateApiCallback) {
        SDKPreconditions.checkNotNull(managerData, "Manager data cannot be null");

        managersApi().updateManager(managerData, updateApiCallback);
    }

    /**
     * Create manager.
     *
     * @param manager the manager
     * @return the http result
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IManager> createManager(final ManagerData manager) throws ClientException {
        SDKPreconditions.checkNotNull(manager, "Manager data is null");
        return managersApi().createManager(manager);
    }

    @Override
    public void createManager(ManagerData managerData, ResultApiCallback<IManager> apiCallback) {
        SDKPreconditions.checkNotNull(managerData, "Manager data is null");
        managersApi().createManager(managerData, apiCallback);
    }

    @Override
    public HttpResult<IManager> getManager(UUID managerId, SDKOptional<ETag> eTagOptional) throws ClientException {
        SDKPreconditions.checkNotNull(managerId, "Manager Id is null");
        RequestValidator.validateETag(eTagOptional);

        return managersApi().getManager(managerId, eTagOptional);
    }

    @Override
    public void getManager(UUID managerId, ResultApiCallback<IManager> apiCallback) {
        getManager(managerId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<IManager> getManager(UUID managerId) throws ClientException {
        return getManager(managerId, SDKOptional.<ETag>absent());
    }

    @Override
    public void getManager(UUID managerId, SDKOptional<ETag> etag, ResultApiCallback<IManager> apiCallback) {
        SDKPreconditions.checkNotNull(managerId, "Manager id is null.");
        RequestValidator.validateETag(etag);

        managersApi().getManager(managerId, etag, apiCallback);
    }

    /**
     * Assign managers to supervisor.
     *
     * @param supervisorId the supervisor id
     * @param managerIdSet the manager id set
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int assignManagersToSupervisor(UUID supervisorId, Set<UUID> managerIdSet) throws ClientException {
        SDKPreconditions.checkNotNull(supervisorId, "Supervisor Id cannot be null");
        SDKPreconditions.checkNotNullOrEmpty(managerIdSet, "Manager Ids cannot be null");

        return managersApi().assignManagersToSupervisor(supervisorId, managerIdSet);
    }

    @Override
    public void assignManagersToSupervisor(UUID supervisorId, Set<UUID> managerIdSet, UpdateApiCallback apiCallback) {
        SDKPreconditions.checkNotNull(supervisorId, "Supervisor Id cannot be null");
        SDKPreconditions.checkNotNullOrEmpty(managerIdSet, "Manager Ids cannot be null");

        managersApi().assignManagersToSupervisor(supervisorId, managerIdSet, apiCallback);
    }

    /**
     * Gets config.
     *
     * @return the config
     * @throws ClientException the client exception
     */
    public HttpResult<List<IConfig>> listConfigs() throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter("deviceType", DeviceType.BEACON.name())
                .build();

        return listConfigs(requestDescription);
    }

    public void listConfigs(ResultApiCallback<List<IConfig>> apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter("deviceType", DeviceType.BEACON.name())
                .build();

        listConfigs(requestDescription, apiCallback);
    }

    @Override
    public void listConfigs(RequestDescription requestDescription, ResultApiCallback<List<IConfig>> apiCallback) {
        RequestValidator.configsListingPreconditions(requestDescription);
        configurationApi().listConfigs(requestDescription, apiCallback);
    }

    @Override
    public HttpResult<List<IConfig>> listConfigs(final RequestDescription requestDescription) throws ClientException {
        RequestValidator.configsListingPreconditions(requestDescription);
        return configurationApi().listConfigs(requestDescription);
    }

    @Override
    public HttpResult<List<ICloudConfig>> listCloudConfigs() throws ClientException {
        return listCloudConfigs(RequestDescription.start()
                .addParameter("deviceType", DeviceType.CLOUD_BEACON.name())
                .build());
    }

    @Override
    public void listCloudConfigs(ResultApiCallback<List<ICloudConfig>> apiCallback) {
        listCloudConfigs(AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public HttpResult<List<ICloudConfig>> listCloudConfigs(RequestDescription requestDescription) throws ClientException {
        RequestValidator.cloudConfigsListingPreconditions(requestDescription);
        return configurationApi().listCloudConfigs(requestDescription);
    }

    @Override
    public void listCloudConfigs(RequestDescription requestDescription, ResultApiCallback<List<ICloudConfig>> apiCallback) {
        RequestValidator.cloudConfigsListingPreconditions(requestDescription);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");

        configurationApi().listCloudConfigs(requestDescription, apiCallback);
    }

    /**
     * Create config.
     *
     * @return the http result
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IConfig> createConfig(ConfigData configData) throws ClientException {
        SDKPreconditions.checkNotNull(configData, "Config data is null.");

        return configurationApi().createConfig(configData);
    }

    @Override
    public void createConfig(ConfigData configData, ResultApiCallback<IConfig> apiCallback) {
        SDKPreconditions.checkNotNull(configData, "Config data is null.");

        configurationApi().createConfig(configData, apiCallback);
    }

    @Override
    public void createCloudConfig(ConfigData configData, ResultApiCallback<ICloudConfig> apiCallback) {
        SDKPreconditions.checkNotNull(configData, "Config data is null.");

        configurationApi().createCloudConfig(configData, apiCallback);
    }

    @Override
    public HttpResult<ICloudConfig> createCloudConfig(ConfigData configData) throws ClientException {
        SDKPreconditions.checkNotNull(configData, "Config data is null.");

        return configurationApi().createCloudConfig(configData);
    }


    /**
     * Apply config.
     *
     * @param config the config
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int applyConfig(IConfig config) throws ClientException {
        SDKPreconditions.checkNotNull(config, "Config is null");
        return devicesApi().applyConfig(config);
    }

    @Override
    public void applyConfig(IConfig config, UpdateApiCallback apiCallback) {
        SDKPreconditions.checkNotNull(config, "Config is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        devicesApi().applyConfig(config, apiCallback);
    }

    @Override
    public int applyCloudConfig(ICloudConfig cloudConfig) throws ClientException {
        SDKPreconditions.checkNotNull(cloudConfig, "Config is null");
        return devicesApi().applyCloudConfig(cloudConfig);
    }

    @Override
    public void applyConfig(ICloudConfig cloudConfig, UpdateApiCallback apiCallback) {
        SDKPreconditions.checkNotNull(cloudConfig, "Config is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        devicesApi().applyConfig(cloudConfig, apiCallback);
    }

    /**
     * Gets config for beacon.
     *
     * @param beaconUniqueId the beacon unique id
     * @return the config for beacon
     * @throws ClientException the client exception
     */
    public HttpResult<IConfig> getConfigForDevice(String beaconUniqueId) throws ClientException {
        return getConfigForDevice(beaconUniqueId, SDKOptional.<ETag>absent());
    }

    public void getConfigForDevice(String beaconUniqueId, ResultApiCallback<IConfig> apiCallback) {
        getConfigForDevice(beaconUniqueId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getConfigForDevice(String beaconUniqueId, SDKOptional<ETag> etag, ResultApiCallback<IConfig> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueId, "Device Unique id is null or empty");
        RequestValidator.validateETag(etag);

        configurationApi().getConfigForDevice(beaconUniqueId, etag, apiCallback);
    }

    /**
     * Gets config for beacon.
     *
     * @param beaconUniqueId the beacon unique id
     * @param eTag           the e tag
     * @return the config for beacon
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IConfig> getConfigForDevice(String beaconUniqueId, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueId, "Beacon unique Id is null or empty");
        RequestValidator.validateETag(eTag);

        return configurationApi().getConfigForDevice(beaconUniqueId, eTag);
    }

    /**
     * Gets cloud config for device.
     *
     * @param deviceUniqueId the device unique id
     * @return the cloud config for device
     * @throws ClientException the client exception
     */
    public HttpResult<ICloudConfig> getCloudConfigForDevice(final String deviceUniqueId) throws ClientException {
        return getCloudConfigForDevice(deviceUniqueId, SDKOptional.<ETag>absent());
    }

    @Override
    public HttpResult<ICloudConfig> getCloudConfigForDevice(String beaconUniqueId, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueId, "Beacon Unique Id is null or empty");
        RequestValidator.validateETag(eTag);

        return configurationApi().getCloudConfigForDevice(beaconUniqueId, eTag);
    }

    public void getCloudConfigForDevice(String beaconUniqueId, ResultApiCallback<ICloudConfig> apiCallback) {
        getCloudConfigForDevice(beaconUniqueId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getCloudConfigForDevice(String beaconUniqueId, SDKOptional<ETag> eTag, ResultApiCallback<ICloudConfig> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueId, "Beacon Unique Id is null or empty");
        RequestValidator.validateETag(eTag);

        configurationApi().getCloudConfigForDevice(beaconUniqueId, eTag, apiCallback);
    }

    /**
     * Gets profiles.
     *
     * @return the profiles
     * @throws ClientException the client exception
     */
    public HttpResult<List<IPreset>> getPresets() throws ClientException {
        return getPresets(SDKOptional.<ETag>absent());
    }

    public void getPresets(ResultApiCallback<List<IPreset>> apiCallback) {
        getPresets(SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getPresets(SDKOptional<ETag> eTag, ResultApiCallback<List<IPreset>> apiCallback) {
        RequestValidator.validateETag(eTag);
        SDKPreconditions.checkNotNull(apiCallback, "Api callback is null.");

        configurationApi().getPresets(eTag, apiCallback);
    }

    /**
     * Gets profiles.
     *
     * @param eTag the e tag
     * @return the profiles
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<List<IPreset>> getPresets(SDKOptional<ETag> eTag) throws ClientException {
        RequestValidator.validateETag(eTag);

        return configurationApi().getPresets(eTag);
    }

    /**
     * Gets profile.
     *
     * @param profileName the profile name
     * @return the profile
     * @throws ClientException the client exception
     */
    public HttpResult<IPreset> getPreset(String profileName) throws ClientException {
        return getPreset(profileName, SDKOptional.<ETag>absent());
    }

    public void getPreset(String profileName,
                          ResultApiCallback<IPreset> apiCallback) {
        configurationApi().getPreset(profileName, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getPreset(String profileName,
                          SDKOptional<ETag> etag,
                          ResultApiCallback<IPreset> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(profileName, "Preset name is null or empty");
        RequestValidator.validateETag(etag);

        configurationApi().getPreset(profileName, etag, apiCallback);
    }

    /**
     * Gets profile.
     *
     * @param profileName the profile name
     * @param eTag        the e tag
     * @return the profile
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IPreset> getPreset(String profileName, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(profileName, "Preset name is null or empty");
        RequestValidator.validateETag(eTag);

        return configurationApi().getPreset(profileName, eTag);
    }

    @Override
    public HttpResult<List<SecureSingleConfig>> listSecureConfigs(SDKOptional<ETag> etag) throws ClientException {
        return configurationApi().listSecureConfigs(etag);
    }

    @Override
    public void listSecureConfigs(SDKOptional<ETag> etag, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        configurationApi().listSecureConfigs(etag, apiCallback);
    }

    @Override
    public HttpResult<List<SecureSingleConfig>> listSecureConfigs(RequestDescription requestDescription) throws ClientException {
        return configurationApi().listSecureConfigs(requestDescription);
    }

    @Override
    public void listSecureConfigs(RequestDescription requestDescription, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        configurationApi().listSecureConfigs(requestDescription, apiCallback);
    }

    @Override
    public void listSecureConfigs(Collection<String> uniqueId, SDKOptional<ETag> eTag, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        configurationApi().listSecureConfigs(uniqueId, eTag, apiCallback);
    }

    @Override
    public HttpResult<List<SecureSingleConfig>> listSecureConfigs(Collection<String> uniqueId, SDKOptional<ETag> eTag) throws ClientException {
        return configurationApi().listSecureConfigs(uniqueId, eTag);
    }

    @Override
    public HttpResult<List<SecureSingleConfig>> applySecureConfig(Collection<SecureSingleConfig> secureConfigApplies) throws ClientException {
        return devicesApi().applySecureConfig(secureConfigApplies);
    }

    @Override
    public void applySecureConfig(Collection<SecureSingleConfig> secureConfigApplies, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        devicesApi().applySecureConfig(secureConfigApplies, apiCallback);
    }

    /**
     * Gets latest firmware for beacons.
     *
     * @param beaconUniqueIds the beacon unique ids
     * @return the latest firmware for beacons
     * @throws ClientException the client exception
     */
    @Deprecated
    @Override
    public HttpResult<Map<String, IFirmware>> getLatestFirmwareForBeacons(final Set<String> beaconUniqueIds) throws ClientException {
        return getLatestFirmwareForBeacons(beaconUniqueIds, SDKOptional.<ETag>absent());
    }

    @Deprecated
    @Override
    public void getLatestFirmwareForBeacons(Set<String> deviceUniqueIds, ResultApiCallback<Map<String, IFirmware>> callback) {
        getLatestFirmwareForBeacons(deviceUniqueIds, SDKOptional.<ETag>absent(), callback);
    }

    @Override
    public void fetchFirmwareFileData(String firmwareName, ResultApiCallback<FileData> resultApiCallback) {
        fetchFirmwareFileData(firmwareName, SDKOptional.<ETag>absent(), resultApiCallback);
    }

    @Override
    public void getFirmware(String firmwareName, DeviceType deviceType, ResultApiCallback<IFirmware> apiCallback) {
        getFirmware(firmwareName, deviceType, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<List<IFirmware>> getFirmwares(Set<String> deviceUniqueIds) throws ClientException {
        return firmwareApi().getFirmwares(deviceUniqueIds);
    }

    @Override
    public void getFirmwares(Set<String> deviceUniqueIds, ResultApiCallback<List<IFirmware>> apiCallback) {
        firmwareApi().getFirmwares(deviceUniqueIds, apiCallback);
    }


    @Deprecated
    @Override
    public HttpResult<Map<String, IFirmware>> getLatestFirmwareForBeacons(Set<String> beaconUniqueIds, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueIds, "Beacon Unique Ids set is null or empty.");

        return firmwareApi().getLatestFirmwareForBeacons(beaconUniqueIds, eTag);
    }

    @Deprecated
    @Override
    public void getLatestFirmwareForBeacons(Set<String> deviceUniqueIds, SDKOptional<ETag> eTag, ResultApiCallback<Map<String, IFirmware>> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(deviceUniqueIds, "Beacon Unique Ids set is null or empty.");
        RequestValidator.validateETag(eTag);

        firmwareApi().getLatestFirmwareForBeacons(deviceUniqueIds, eTag, apiCallback);
    }

    /**
     * Gets firmware.
     *
     * @param firmwareName the firmware name
     * @return the firmware
     * @throws ClientException the client exception
     */
    public HttpResult<IFirmware> getFirmware(final String firmwareName, DeviceType deviceType) throws ClientException {
        return getFirmware(firmwareName, deviceType, SDKOptional.<ETag>absent());
    }

    /**
     * Gets firmware.
     *
     * @param firmwareName the firmware name
     * @param eTag         the e tag
     * @return the firmware
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IFirmware> getFirmware(String firmwareName, DeviceType deviceType, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(firmwareName, "Firmware name is null or empty");
        SDKPreconditions.checkNotNull(deviceType, "Device type is null");
        RequestValidator.validateETag(eTag);

        return firmwareApi().getFirmware(firmwareName, deviceType, eTag);
    }

    @Override
    public void getFirmware(String firmwareName, DeviceType deviceType, SDKOptional<ETag> eTag, ResultApiCallback<IFirmware> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(firmwareName, "Firmware name is null or empty");
        SDKPreconditions.checkNotNull(deviceType, "Device type is null");
        RequestValidator.validateETag(eTag);

        firmwareApi().getFirmware(firmwareName, deviceType, eTag, apiCallback);
    }

    /**
     * Fetch firmware file data.
     *
     * @param firmwareName the firmware name
     * @return the http result
     * @throws ClientException the client exception
     */
    public HttpResult<FileData> fetchFirmwareFileData(final String firmwareName) throws ClientException {
        return fetchFirmwareFileData(firmwareName, SDKOptional.<ETag>absent());
    }

    /**
     * Fetch firmware file data.
     *
     * @param firmwareName the firmware name
     * @param eTag         the e tag
     * @return the http result
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<FileData> fetchFirmwareFileData(String firmwareName, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(firmwareName, "Firmware name is null or empty.");
        RequestValidator.validateETag(eTag);

        return firmwareApi().fetchFirmwareFileData(firmwareName, eTag);
    }

    @Override
    public void fetchFirmwareFileData(String firmwareName, SDKOptional<ETag> etag, ResultApiCallback<FileData> resultApiCallback) {
        SDKPreconditions.checkNotNull(firmwareName, "Firmware name is null");
        RequestValidator.validateETag(etag);

        firmwareApi().fetchFirmwareFileData(firmwareName, etag, resultApiCallback);
    }

    @Override
    public HttpResult<IDevice> getDeviceByNamespaceAndInstanceId(String namespace, String instanceId) throws ClientException {
        SDKPreconditions.checkNotNull(namespace, "Namespace is null.");
        SDKPreconditions.checkNotNull(instanceId, "Instance id is null.");
        return devicesApi().getDeviceByNamespaceAndInstanceId(namespace, instanceId);
    }

    @Override
    public HttpResult<IDevice> getDeviceByNamespaceAndInstanceId(String namespace, String instanceId, SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNull(namespace, "Namespace is null.");
        SDKPreconditions.checkNotNull(instanceId, "Instance id is null.");
        RequestValidator.validateETag(eTag);
        return devicesApi().getDeviceByNamespaceAndInstanceId(namespace, instanceId, eTag);
    }

    @Override
    public void getDeviceByNamespaceAndInstanceId(String namespace, String instanceId, ResultApiCallback<IDevice> apiCallback) {
        getDeviceByNamespaceAndInstanceId(namespace, instanceId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getDeviceByNamespaceAndInstanceId(String namespace, String instanceId, SDKOptional<ETag> eTag, ResultApiCallback<IDevice> apiCallback) {
        SDKPreconditions.checkNotNull(namespace, "Namespace is null.");
        SDKPreconditions.checkNotNull(instanceId, "Instance id is null.");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        RequestValidator.validateETag(eTag);
        devicesApi().getDeviceByNamespaceAndInstanceId(namespace, instanceId, eTag, apiCallback);
    }

    /**
     * Gets action content.
     *
     * @param actionId the action id
     * @return the action content
     * @throws ClientException the client exception
     */
    public HttpResult<FileData> getActionContent(final UUID actionId) throws ClientException {
        return getActionContent(actionId, SDKOptional.<ETag>absent());
    }

    /**
     * Gets action content.
     *
     * @param actionId the action id
     * @param eTag     the e tag
     * @return the action content
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<FileData> getActionContent(UUID actionId, final SDKOptional<ETag> eTag) throws ClientException {
        SDKPreconditions.checkNotNull(actionId, "Action Id is null.");

        return actionsApi().getActionContent(actionId, eTag);
    }


    @Override
    public HttpResult<IAction> getAction(UUID actionId) throws ClientException {
        return getAction(actionId, SDKOptional.<ETag>absent());
    }

    /**
     * Creates Content Action.
     *
     * @param actionData the actionData
     * @return the http result
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IContentAction> createContentAction(final ActionData actionData, final File file) throws ClientException {
        RequestValidator.contentActionCreationPreconditions(actionData, file);

        return actionsApi().createContentAction(actionData, file);
    }

    @Override
    public void createContentAction(ActionData actionData, File contentFile, ResultApiCallback<IContentAction> apiCallback) {
        RequestValidator.contentActionCreationPreconditions(actionData, contentFile);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");

        actionsApi().createContentAction(actionData, contentFile, apiCallback);
    }

    /**
     * Creates Browser Action.
     *
     * @param actionData the actionData
     * @return the http result
     * @throws ClientException the client exception
     */
    @Override
    public HttpResult<IBrowserAction> createBrowserAction(ActionData actionData) throws ClientException {
        RequestValidator.browserActionCreationPreconditions(actionData);

        return actionsApi().createBrowserAction(actionData);
    }

    @Override
    public void createBrowserAction(ActionData actionData, ResultApiCallback<IBrowserAction> resultApiCallback) {
        RequestValidator.browserActionCreationPreconditions(actionData);
        SDKPreconditions.checkNotNull(resultApiCallback, "Callback is null.");

        actionsApi().createBrowserAction(actionData, resultApiCallback);
    }

    @Override
    public void getAction(UUID actionId, ResultApiCallback<IAction> resultApiCallback) {
        getAction(actionId, SDKOptional.<ETag>absent(), resultApiCallback);
    }

    @Override
    public HttpResult<IAction> getAction(UUID actionId, SDKOptional<ETag> eTag) throws ClientException {
        RequestValidator.actionRetrievalPreconditions(actionId, eTag);

        return actionsApi().getAction(actionId, eTag);
    }

    @Override
    public void getAction(UUID actionId, SDKOptional<ETag> etag, ResultApiCallback<IAction> resultApiCallback) {
        RequestValidator.actionRetrievalPreconditions(actionId, etag);
        SDKPreconditions.checkNotNull(resultApiCallback, "Callback is null.");

        actionsApi().getAction(actionId, etag, resultApiCallback);
    }

    /**
     * Update action.
     *
     * @param actionId the action id
     * @param file     the file
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int updateAction(final UUID actionId, final File file) throws ClientException {
        RequestValidator.actionUpdatePreconditions(actionId, file);
        return actionsApi().updateAction(actionId, file);
    }

    @Override
    public void updateAction(UUID actionId, File file, UpdateApiCallback callback) {
        RequestValidator.actionUpdatePreconditions(actionId, file);
        SDKPreconditions.checkNotNull(callback, "Callback is null.");

        actionsApi().updateAction(actionId, file, callback);
    }

    @Override
    public int updateAction(UUID actionId, String url) throws ClientException {
        RequestValidator.actionUpdatePreconditions(actionId, url);
        return actionsApi().updateAction(actionId, url);
    }

    @Override
    public void updateAction(UUID actionId, String url, UpdateApiCallback callback) {
        RequestValidator.actionUpdatePreconditions(actionId, url);
        SDKPreconditions.checkNotNull(callback, "Callback is null.");
        actionsApi().updateAction(actionId, url, callback);
    }

    /**
     * Delete action.
     *
     * @param actionId the action id
     * @return the int
     * @throws ClientException the client exception
     */
    @Override
    public int deleteAction(UUID actionId) throws ClientException {
        RequestValidator.actionDeletionPreconditions(actionId);
        return actionsApi().deleteAction(actionId);
    }

    @Override
    public void deleteAction(UUID actionId, UpdateApiCallback apiCallback) {
        RequestValidator.actionDeletionPreconditions(actionId);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");

        actionsApi().deleteAction(actionId, apiCallback);
    }

    @Override
    public void getActionContent(UUID actionId, SDKOptional<ETag> eTagSDKOptional, ResultApiCallback<FileData> resultApiCallback) {
        SDKPreconditions.checkNotNull(actionId, "Action Id is null.");
        RequestValidator.validateETag(eTagSDKOptional);

        actionsApi().getActionContent(actionId, eTagSDKOptional, resultApiCallback);
    }

    @Override
    public HttpResult<List<IAction>> getActionsForDevice(String deviceUniqueId) throws ClientException {
        SDKPreconditions.checkNotNull(deviceUniqueId, "Device unique id is null.");
        return actionsApi().getActionsForDevice(deviceUniqueId);
    }

    @Override
    public void getActionsForDevice(String deviceUniqueId, ResultApiCallback<List<IAction>> resultApiCallback) {
        SDKPreconditions.checkNotNull(deviceUniqueId, "Device unique id is null.");
        actionsApi().getActionsForDevice(deviceUniqueId, resultApiCallback);
    }

    /**
     * Gets proximities.
     *
     * @return the proximities
     * @throws ClientException the client exception
     */
    public HttpResult<List<IProximityUUID>> listProximities() throws ClientException {
        return commonsApi().listProximities();
    }

    @Override
    public void listProximities(SDKOptional<ETag> etag, ResultApiCallback<List<IProximityUUID>> apiCallback) {
        RequestValidator.validateETag(etag);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        commonsApi().listProximities(etag, apiCallback);
    }

    @Override
    public void listProximities(ResultApiCallback<List<IProximityUUID>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        commonsApi().listProximities(apiCallback);
    }

    @Override
    public HttpResult<List<IProximityUUID>> listProximities(SDKOptional<ETag> eTag) throws ClientException {
        RequestValidator.validateETag(eTag);
        return commonsApi().listProximities(eTag);
    }


    @Override
    public HttpResult<List<IProximityUUID>> resolveProximities(List<UUID> proximities) throws ClientException {
        SDKPreconditions.checkNotNull(proximities, "Provided proximities list is null");
        return commonsApi().resolveProximities(proximities);
    }

    @Override
    public HttpResult<List<IProximityUUID>> resolveProximities(List<UUID> proximities, SDKOptional<ETag> etag) throws ClientException {
        SDKPreconditions.checkNotNull(proximities, "Provided proximities list is null");
        RequestValidator.validateETag(etag);
        return commonsApi().resolveProximities(proximities, etag);
    }

    @Override
    public void resolveProximities(List<UUID> proximities, ResultApiCallback<List<IProximityUUID>> apiCallback) {
        SDKPreconditions.checkNotNull(proximities, "Provided proximities list is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        commonsApi().resolveProximities(proximities, apiCallback);
    }

    @Override
    public void resolveProximities(List<UUID> proximities, SDKOptional<ETag> etag, ResultApiCallback<List<IProximityUUID>> apiCallback) {
        SDKPreconditions.checkNotNull(proximities, "Provided proximities list is null");
        RequestValidator.validateETag(etag);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        commonsApi().resolveProximities(proximities, etag, apiCallback);
    }

    @Override
    public HttpResult<List<INamespace>> listNamespaces() throws ClientException {
        return commonsApi().listNamespaces();
    }

    @Override
    public HttpResult<List<INamespace>> listNamespaces(SDKOptional<ETag> etag) throws ClientException {
        RequestValidator.validateETag(etag);
        return commonsApi().listNamespaces(etag);
    }

    @Override
    public void listNamespaces(ResultApiCallback<List<INamespace>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        commonsApi().listNamespaces(apiCallback);
    }

    @Override
    public void listNamespaces(SDKOptional<ETag> etag, ResultApiCallback<List<INamespace>> apiCallback) {
        RequestValidator.validateETag(etag);
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        commonsApi().listNamespaces(etag, apiCallback);
    }

    @Override
    public HttpResult<List<INamespace>> resolveNamespaces(List<String> namespaces) throws ClientException {
        SDKPreconditions.checkNotNull(namespaces, "Provided namespaces list is null");
        return commonsApi().resolveNamespaces(namespaces);
    }

    @Override
    public HttpResult<List<INamespace>> resolveNamespaces(List<String> namespaces, SDKOptional<ETag> etag) throws ClientException {
        SDKPreconditions.checkNotNull(namespaces, "Provided namespaces list is null");
        RequestValidator.validateETag(etag);
        return commonsApi().resolveNamespaces(namespaces, etag);
    }

    @Override
    public void resolveNamespaces(List<String> namespaces, ResultApiCallback<List<INamespace>> apiCallback) {
        SDKPreconditions.checkNotNull(namespaces, "Provided namespaces list is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        commonsApi().resolveNamespaces(namespaces, apiCallback);
    }

    @Override
    public void resolveNamespaces(List<String> namespaces, SDKOptional<ETag> etag, ResultApiCallback<List<INamespace>> apiCallback) {
        SDKPreconditions.checkNotNull(namespaces, "Provided namespaces list is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        RequestValidator.validateETag(etag);
        commonsApi().resolveNamespaces(namespaces, etag, apiCallback);
    }

    @Override
    public HttpResult<List<IVenue>> listVenues() throws ClientException {
        return venuesApi().listVenues(AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public HttpResult<List<IVenue>> listVenues(RequestDescription requestDescription) throws ClientException {
        SDKPreconditions.checkNotNull(requestDescription, "RequestDescription is null.");
        return venuesApi().listVenues(requestDescription);
    }

    @Override
    public void listVenues(ResultApiCallback<List<IVenue>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        listVenues(AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public void listVenues(RequestDescription requestDescription, ResultApiCallback<List<IVenue>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        SDKPreconditions.checkNotNull(requestDescription, "RequestDescription is null.");
        venuesApi().listVenues(requestDescription, apiCallback);
    }


    @Override
    public HttpResult<List<IDevice>> listDevices() throws ClientException {
        return devicesApi().listDevices();
    }

    @Override
    public HttpResult<List<IDevice>> listDevices(RequestDescription requestDescription) throws ClientException {
        SDKPreconditions.checkNotNull(requestDescription, "RequestDescription is null.");
        return devicesApi().listDevices(requestDescription);
    }

    @Override
    public void listDevices(ResultApiCallback<List<IDevice>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        devicesApi().listDevices(apiCallback);
    }

    @Override
    public void listDevices(RequestDescription requestDescription, ResultApiCallback<List<IDevice>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        SDKPreconditions.checkNotNull(requestDescription, "RequestDescription is null.");
        devicesApi().listDevices(requestDescription, apiCallback);
    }

    @Override
    public HttpResult<List<IDevice>> getDevicesByProximity(UUID proximityUUID, int major, int minor) throws ClientException {
        return devicesApi().getDevicesByProximity(proximityUUID, major, minor, SDKOptional.<ETag>absent());
    }

    @Override
    public HttpResult<List<IDevice>> getDevicesByProximity(UUID proximityUUID, int major, int minor, SDKOptional<ETag> eTag) throws ClientException {
        return devicesApi().getDevicesByProximity(proximityUUID, major, minor, eTag);
    }

    @Override
    public void getDevicesByProximity(UUID proximityUUID, int major, int minor, ResultApiCallback<List<IDevice>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        devicesApi().getDevicesByProximity(proximityUUID, major, minor, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public void getDevicesByProximity(UUID proximityUUID, int major, int minor, SDKOptional<ETag> eTag, ResultApiCallback<List<IDevice>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        devicesApi().getDevicesByProximity(proximityUUID, major, minor, eTag, apiCallback);
    }

    @Override
    public HttpResult<List<IManager>> listManagers() throws ClientException {
        return managersApi().listManagers(AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public HttpResult<List<IManager>> listManagers(RequestDescription requestDescription) throws ClientException {
        SDKPreconditions.checkNotNull(requestDescription, "request description is null.");
        return managersApi().listManagers(requestDescription);
    }

    @Override
    public void listManagers(ResultApiCallback<List<IManager>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        listManagers(AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }


    @Override
    public void listManagers(RequestDescription requestDescription, ResultApiCallback<List<IManager>> apiCallback) {
        SDKPreconditions.checkNotNull(requestDescription, "request description is null.");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");

        managersApi().listManagers(requestDescription, apiCallback);
    }

    @Override
    public HttpResult<List<IDevice>> listUnassignedDevicesForManager(final UUID managerId) throws ClientException {
        return listUnassignedDevicesForManager(managerId, AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public void listUnassignedDevicesForManager(UUID managerId, ResultApiCallback<List<IDevice>> apiCallback) {
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        listUnassignedDevicesForManager(managerId, AbstractApiAccessor.DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public void listUnassignedDevicesForManager(UUID managerId, RequestDescription requestDescription, ResultApiCallback<List<IDevice>> apiCallback) {
        SDKPreconditions.checkNotNull(managerId, "Manager id is null");
        SDKPreconditions.checkNotNull(requestDescription, "request description is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null.");
        devicesApi().listUnassignedDevicesForManager(managerId, requestDescription, apiCallback);
    }

    @Override
    public HttpResult<List<IDevice>> listUnassignedDevicesForManager(UUID managerId, RequestDescription requestDescription) throws ClientException {
        SDKPreconditions.checkNotNull(managerId, "Manager Id is null");
        SDKPreconditions.checkNotNull(requestDescription, "Request description is null");

        return devicesApi().listUnassignedDevicesForManager(managerId, requestDescription);
    }


    @Override
    public void resolveIBeacon(Collection<BeaconId> beaconIds, SDKOptional<ETag> eTag, ResultApiCallback<List<IBeaconFutureId>> apiCallback) {
        SDKPreconditions.checkNotNull(beaconIds, "Provided beaconid list is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null");
        devicesApi().resolveIBeacon(beaconIds, eTag, apiCallback);
    }

    @Override
    public HttpResult<List<IBeaconFutureId>> resolveIBeacon(Collection<BeaconId> beaconIds, SDKOptional<ETag> eTagSDKOptional) throws ClientException {
        SDKPreconditions.checkNotNull(beaconIds, "Provided beaconid list is null");
        return devicesApi().resolveIBeacon(beaconIds, eTagSDKOptional);
    }

    @Override
    public void resolveEddystone(Collection<EddystoneUID> eddystoneUids, SDKOptional<ETag> eTag, ResultApiCallback<List<EddystoneFutureUID>> apiCallback) {
        SDKPreconditions.checkNotNull(eddystoneUids, "Provided eddystone uids list is null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback is null");
        devicesApi().resolveEddystone(eddystoneUids, eTag, apiCallback);
    }

    @Override
    public HttpResult<List<EddystoneFutureUID>> resolveEddystone(Collection<EddystoneUID> eddystoneUids, SDKOptional<ETag> eTagSDKOptional) throws ClientException {
        SDKPreconditions.checkNotNull(eddystoneUids, "Provided eddystone uids list is null");
        return devicesApi().resolveEddystone(eddystoneUids, eTagSDKOptional);
    }

    @Override
    public int sendEvents(EventPacket packet) throws ClientException {
        SDKPreconditions.checkNotNull(packet, "Provided event packet is null");
        return commonsApi().sendEvents(packet);
    }

    @Override
    public HttpResult<List<SecureCommandResponse>> getSecureCommand(Collection<String> beaconUniqueIds, SecureCommandType secureCommandType) throws ClientException {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueIds, "Beacon ids cannot be null or empty");
        SDKPreconditions.checkNotNull(secureCommandType, "Command type cannot be null");
        SDKPreconditions.checkAllowedSize(beaconUniqueIds, 100, "Beacon ids count is greater than 100");
        return commandApi().getSecureCommand(beaconUniqueIds, secureCommandType);
    }

    @Override
    public void getSecureCommand(Collection<String> beaconUniqueIds, SecureCommandType secureCommandType, ResultApiCallback<List<SecureCommandResponse>> apiCallback) {
        SDKPreconditions.checkNotNullOrEmpty(beaconUniqueIds, "Beacon ids cannot be null or empty");
        SDKPreconditions.checkNotNull(secureCommandType, "Command type cannot be null");
        SDKPreconditions.checkNotNull(apiCallback, "Callback cannot be null");
        SDKPreconditions.checkAllowedSize(beaconUniqueIds, 100, "Beacon ids count is greater than 100");
        commandApi().getSecureCommand(beaconUniqueIds, secureCommandType, apiCallback);
    }
}
