package com.kontakt.sdk.android.http;

import android.net.Uri;
import android.os.Build;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.interfaces.SDKPredicate;
import com.kontakt.sdk.android.common.interfaces.SDKThrowableFunction;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.Closeables;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.data.AbstractEntityData;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.kontakt.sdk.android.http.interfaces.UpdateApiCallback;
import com.squareup.okhttp.Callback;
import com.squareup.okhttp.FormEncodingBuilder;
import com.squareup.okhttp.Headers;
import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;
import com.squareup.okhttp.Response;
import com.squareup.okhttp.ResponseBody;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * AbstractApiClientDelegate provides abstraction for multiple API Client delegates.
 * There is an associated ApiClientDelegate for each URI prefix (beacon, venue, action etc.).
 */
abstract class AbstractApiAccessor implements Closeable {

    private static final String TAG = AbstractApiAccessor.class.getSimpleName();

    static final RequestDescription DEFAULT_REQUEST_DESCRIPTION = RequestDescription.start().build();

    static final MediaType MEDIA_TYPE_APPLICATION_FORM_URLENCODED = MediaType.parse("application/x-www-form-urlencoded");

    static final MediaType MEDIA_TYPE_JSON = MediaType.parse("application/json; charset=utf-8");

    static final SDKThrowableFunction<ResponseBody, JSONObject, Exception> JSON_OBJECT_EXTRACT_FUNCTION = new SDKThrowableFunction<ResponseBody, JSONObject, Exception>() {
        @Override
        public JSONObject apply(ResponseBody object) throws Exception {
            return new JSONObject(object.string());
        }
    };

    static final SDKThrowableFunction<ResponseBody, byte[], Exception> BYTE_ARRAY_EXTRACT_FUNCTION = new SDKThrowableFunction<ResponseBody, byte[], Exception>() {
        @Override
        public byte[] apply(ResponseBody object) throws Exception {
            return object.bytes();
        }
    };

    static final SDKThrowableFunction<ResponseBody, JSONArray, Exception> JSON_ARRAY_EXTRACT_FUNCTION = new SDKThrowableFunction<ResponseBody, JSONArray, Exception>() {
        @Override
        public JSONArray apply(ResponseBody object) throws Exception {
            return new JSONArray(object.string());
        }
    };

    private static final Headers EMPTY_HEADERS = new Headers.Builder().build();

    private final OkHttpClient httpClient;

    private final Headers standardHeaders;

    private final Uri baseUri;

    /**
     * Instantiates a new Abstract api client delegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    protected AbstractApiAccessor(String apiKey,
                                  String apiUrl) {
        this.httpClient = new OkHttpClient();
        this.standardHeaders = new Headers.Builder()
                .add(ApiConstants.MainHeaders.API_KEY, apiKey)
                .add(ApiConstants.MainHeaders.ACCEPT, String.format(ApiConstants.MainHeaders.ACCEPT_VND, KontaktApiClient.ACCEPT_VERSION))
                .add(ApiConstants.MainHeaders.KONTAKT_AGENT, String.format(ApiConstants.MainHeaders.KONTAKT_AGENT_ANDROID, KontaktApiClient.ACCEPT_VERSION, Build.VERSION.SDK_INT))
                .build();
        this.baseUri = Uri.parse(apiUrl);
    }

    @Override
    public void close() throws IOException {

    }

    /**
     * Transforms Http Response's body to {@link org.json.JSONObject} and then converts to
     * desired type.
     *
     * @param <T>                       the type parameter
     * @param response                  the response
     * @param expectedSuccessHttpStatus the expected success http status
     * @param buildFunction             the build function
     * @return the result
     * @throws ClientException the client exception
     */
    protected static <T> HttpResult<T> transformJSONToEntity(final Response response,
                                                             final int expectedSuccessHttpStatus,
                                                             final SDKFunction<JSONObject, T> buildFunction) throws ClientException {
        return FluentResponse.<T>of(response, expectedSuccessHttpStatus)
                .transformToJSON()
                .toSingleEntity(buildFunction);
    }

    /**
     * Transform HttpResponse with JSON content to list.
     *
     * @param <T>                       the type parameter
     * @param httpResponse              the http response
     * @param expectedSuccessHttpStatus the expected success http status
     * @param jsonEntry                 the json entry
     * @param buildFunction             the build function
     * @return the http result
     */
    protected static <T> HttpResult<List<T>> transformJSONToList(final Response httpResponse,
                                                                 final int expectedSuccessHttpStatus,
                                                                 final String jsonEntry,
                                                                 final SDKFunction<JSONObject, T> buildFunction) {
        return FluentResponse.<T>of(httpResponse, expectedSuccessHttpStatus)
                .transformToJSON()
                .toList(jsonEntry, buildFunction);
    }

    protected static <T> HttpResult<List<T>> transformJSONToList(final Response response,
                                                                 final int[] expectedSucceddHttpStatuses,
                                                                 final String jsonEntry,
                                                                 final SDKFunction<JSONObject, T> buildFunction) {
        return FluentResponse.<T>of(response, expectedSucceddHttpStatuses)
                .transformToJSON()
                .toList(jsonEntry, buildFunction);
    }

    /**
     * Transforms Http Response's entity to byte array and then converts to desired type.
     *
     * @param <T>                       the type parameter
     * @param response                  the response
     * @param expectedSuccessHttpStatus the expected success http status
     * @param buildFunction             the build function
     * @return the result
     * @throws ClientException the client exception
     */
    protected static <T> HttpResult<T> transformByteArrayToHttpResult(final Response response,
                                                                      final int expectedSuccessHttpStatus,
                                                                      final SDKFunction<byte[], T> buildFunction) throws ClientException {

        return FluentResponse.<T>of(response, expectedSuccessHttpStatus)
                .transformToByteArray()
                .toHttpResult(buildFunction);
    }

    /**
     * Transforms Http Response to desired entity by extracting the content to desired format and then to build the
     * entity from it provided that Http Response status is equal to the expected one.
     *
     * @param <K>                       the generic extract parameter
     * @param <T>                       the generic build parameter
     * @param response                  the response
     * @param expectedSuccessHttpStatus the expected success http status
     * @param extractFunction           the extract function
     * @param buildFunction             the build function
     * @return the result
     * @throws ClientException thrown if transformation fails.
     */
    protected static <K, T> HttpResult<T> transform(final Response response,
                                                    final int[] expectedSuccessHttpStatus,
                                                    final SDKThrowableFunction<ResponseBody, K, Exception> extractFunction,
                                                    final SDKFunction<K, T> buildFunction) throws ClientException {
        ResponseBody body = response.body();
        try {
            final int httpStatus = response.code();
            final HttpResult.Builder<T> builder = new HttpResult.Builder<T>()
                    .setHttpStatus(httpStatus);

            builder.setETagValue(response.header(ETag.HEADER_NAME_RESPONSE));
            boolean validHttpStatusCode = false;
            for (int statusCodeIndex = 0; statusCodeIndex < expectedSuccessHttpStatus.length; statusCodeIndex++) {
                if (httpStatus == expectedSuccessHttpStatus[statusCodeIndex]) {
                    validHttpStatusCode = true;
                }
            }
            if (validHttpStatusCode && response.body().contentLength() > 0) {
                final K extractedObject = extractFunction.apply(body);
                final T builtObject = buildFunction.apply(extractedObject);
                builder.setValue(builtObject);
            }

            return builder.build();
        } catch (Exception e) {
            Logger.e(TAG + " Error ocures during transforming", e);
            throw new ClientException(e);
        } finally {
            try {
                Closeables.close(body, true);
            } catch (IOException ignored) {
            }
        }
    }

    /**
     * Returns http status code from HTTP response and closes it quietly.
     *
     * @param httpResponse the http response
     * @return Http Status code
     */
    protected static int httpStatusCode(final Response httpResponse) {
        try {
            return httpResponse.code();
        } finally {
            try {
                Closeables.close(httpResponse.body(), true);
            } catch (IOException ignored) {
            }
        }
    }

    /**
     * Post request and return HttpResponse.
     *
     * @param uri               the uri
     * @param headers           the headers
     * @param bodyParametersMap the request body
     * @return the http response
     * @throws Exception the exception
     */
    final Response post(final String uri,
                        final Headers headers,
                        final List<Map.Entry<String, String>> bodyParametersMap) throws Exception {

        final Request request = buildPostRequest(uri, headers, bodyParametersMap);

        return httpClient.newCall(request).execute();
    }

    final <T> Response post(final String uri,
                            final RequestDescription requestDescription,
                            final T data,
                            final SDKFunction<T, JSONObject> toJSON) throws ClientException {
        try {
            JSONObject payload = toJSON.apply(data);
            Request request = buildPostRequest(uri, requestDescription.getHeaders(), payload);
            return httpClient.newCall(request).execute();
        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    final <T> int postAndReturnHttpStatus(final String uri,
                                          final RequestDescription requestDescription,
                                          final T data,
                                          final SDKFunction<T, JSONObject> toJSON) throws ClientException {
        Response response = post(uri, requestDescription, data, toJSON);
        return httpStatusCode(response);
    }


    final <T> void postAsyncAndBuildFromJSONObject(final String uri,
                                                   final RequestDescription description,
                                                   final int expectedHttpStatus,
                                                   final ResultApiCallback<T> apiCallback,
                                                   final SDKFunction<JSONObject, T> buildFunction) {
        postAsync(uri,
                description,
                new int[]{expectedHttpStatus},
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                buildFunction);
    }

    final <T> void postAsyncAndBuildFromJSONObject(final String uri,
                                                   final RequestDescription description,
                                                   final int[] expectedHttpStatuses,
                                                   final ResultApiCallback<T> apiCallback,
                                                   final SDKFunction<JSONObject, T> buildFunction) {
        postAsync(uri,
                description,
                expectedHttpStatuses,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                buildFunction);
    }

    final void postAsyncAndReturnHttpStatus(final String uri,
                                            final RequestDescription requestDescription,
                                            final UpdateApiCallback updateApiCallback) {
        final Request request = buildPostRequest(uri,
                requestDescription.getHeaders(),
                requestDescription.getParameters());

        httpClient.newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Request request, IOException e) {
                updateApiCallback.onFailure(new ClientException(e));
            }

            @Override
            public void onResponse(Response response) throws IOException {
                try {
                    updateApiCallback.onSuccess(httpStatusCode(response));
                } catch (Exception e) {
                    updateApiCallback.onFailure(new ClientException(response.code(), e));
                }
            }
        });
    }

    final <K, T> void postAsync(final String uri,
                                final RequestDescription requestDescription,
                                final int expectedHttpStatus,
                                final ResultApiCallback<T> apiCallback,
                                final SDKThrowableFunction<ResponseBody, K, Exception> extractFunction,
                                final SDKFunction<K, T> buildFunction) {
        postAsync(uri,
                requestDescription,
                new int[]{expectedHttpStatus},
                apiCallback,
                extractFunction,
                buildFunction);
    }

    final <K, T> void postAsync(final String uri,
                                final RequestDescription requestDescription,
                                final int[] expectedHttpStatus,
                                final ResultApiCallback<T> apiCallback,
                                final SDKThrowableFunction<ResponseBody, K, Exception> extractFunction,
                                final SDKFunction<K, T> buildFunction) {
        final Request request = buildPostRequest(uri,
                requestDescription.getHeaders(),
                requestDescription.getParameters());

        httpClient.newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Request request, IOException e) {
                apiCallback.onFailure(new ClientException(e));
            }

            @Override
            public void onResponse(Response response) throws IOException {
                try {
                    apiCallback.onSuccess(transform(
                            response,
                            expectedHttpStatus,
                            extractFunction,
                            buildFunction));
                } catch (Exception e) {
                    apiCallback.onFailure(new ClientException(response.code(), e));
                }
            }
        });
    }

    /**
     * Post request and return HttpResponse.
     *
     * @param uri               the uri
     * @param bodyParametersMap the request body parameters map
     * @return the http response
     * @throws Exception the exception
     */
    final Response post(final String uri,
                        final List<Map.Entry<String, String>> bodyParametersMap) throws Exception {
        return post(uri, EMPTY_HEADERS, bodyParametersMap);
    }

    /**
     * Get response.
     *
     * @param uri        the uri
     * @param headers    the headers
     * @param parameters the parameters
     * @return the response
     */
    final Response get(final String uri,
                       final Headers headers,
                       final List<Map.Entry<String, String>> parameters) throws Exception {
        final Request request = buildGetRequest(uri, headers, parameters);
        return httpClient.newCall(request).execute();
    }

    /**
     * Get http response.
     *
     * @param uri     the uri
     * @param headers the headers
     * @return the http response
     * @throws Exception the exception
     */
    final Response get(final String uri, final Headers headers) throws Exception {
        return get(uri, headers, Collections.<Map.Entry<String, String>>emptyList());
    }

    /**
     * Get http response.
     *
     * @param uri the uri
     * @return the http response
     * @throws Exception the exception
     */
    final Response get(final String uri) throws Exception {
        return get(uri, EMPTY_HEADERS);
    }

    final <T> void getAsyncAndRetrieveFromJSONObject(final String uri,
                                                     final RequestDescription desc,
                                                     final int expectedHttpStatus,
                                                     final ResultApiCallback<T> apiCallback,
                                                     final SDKFunction<JSONObject, T> buildFunction) {
        getAsync(uri, desc, expectedHttpStatus, apiCallback, JSON_OBJECT_EXTRACT_FUNCTION, buildFunction);
    }

    final <T> void getAsyncAndRetrieveFromByteArray(final String uri,
                                                    final RequestDescription desc,
                                                    final ResultApiCallback<T> apiCallback,
                                                    final SDKFunction<byte[], T> buildFunction) {
        getAsync(uri,
                desc,
                HttpUtils.SC_OK,
                apiCallback,
                BYTE_ARRAY_EXTRACT_FUNCTION,
                buildFunction);
    }

    final <K, T> void getAsync(final String uri,
                               final RequestDescription requestDescription,
                               final int[] expectedHttpStatus,
                               final ResultApiCallback<T> apiCallback,
                               final SDKThrowableFunction<ResponseBody, K, Exception> extractFunction,
                               final SDKFunction<K, T> buildFunction) {
        final Request request = buildGetRequest(uri, requestDescription.getHeaders(), requestDescription.getParameters());

        httpClient.newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Request request, IOException e) {
                apiCallback.onFailure(new ClientException(e));
            }

            @Override
            public void onResponse(Response response) throws IOException {
                try {
                    apiCallback.onSuccess(transform(
                            response,
                            expectedHttpStatus,
                            extractFunction,
                            buildFunction));
                } catch (Exception e) {
                    apiCallback.onFailure(new ClientException(response.code(), e));
                }
            }
        });
    }

    final <K, T> void getAsync(final String uri,
                               final RequestDescription requestDescription,
                               final int expectedHttpStatus,
                               final ResultApiCallback<T> apiCallback,
                               final SDKThrowableFunction<ResponseBody, K, Exception> extractFunction,
                               final SDKFunction<K, T> buildFunction) {
        getAsync(uri,
                requestDescription,
                new int[]{expectedHttpStatus},
                apiCallback,
                extractFunction,
                buildFunction);
    }

    final <T> void transformToListAsynchronously(final String uri,
                                                 final RequestDescription requestDescription,
                                                 final int expectedHttpStatus,
                                                 final String jsonEntry,
                                                 final ResultApiCallback<List<T>> apiCallback,
                                                 final SDKFunction<JSONObject, T> buildFunction) {
        final Request request = buildGetRequest(uri,
                requestDescription.getHeaders(),
                requestDescription.getParameters());

        httpClient.newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Request request, IOException e) {
                apiCallback.onFailure(new ClientException(e));
            }

            @Override
            public void onResponse(Response response) throws IOException {
                try {
                    apiCallback.onSuccess(transformJSONToList(response,
                            expectedHttpStatus,
                            jsonEntry,
                            buildFunction));
                } catch (Exception e) {
                    apiCallback.onFailure(new ClientException(response.code(), e));
                }
            }
        });
    }

    /**
     * Gets and transform.
     *
     * @param uri                the uri
     * @param requestDescription the request description
     * @param convertFunction    the convert function
     * @return the and transform
     * @throws ClientException the client exception
     */
    protected <T> HttpResult<T> getAndTransform(final String uri,
                                                final RequestDescription requestDescription,
                                                final SDKFunction<JSONObject, T> convertFunction) throws ClientException {
        try {
            final Response httpResponse = get(uri,
                    requestDescription.getHeaders(),
                    requestDescription.getParameters());
            return transformJSONToEntity(httpResponse,
                    HttpUtils.SC_OK,
                    convertFunction);
        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    /**
     * Gets and transform byte array.
     *
     * @param uri                the uri
     * @param requestDescription the request description
     * @param convertFunction    the convert function
     * @return the and transform byte array
     * @throws ClientException the client exception
     */
    protected <T> HttpResult<T> getAndTransformByteArray(final String uri,
                                                         final RequestDescription requestDescription,
                                                         final SDKFunction<byte[], T> convertFunction) throws ClientException {
        try {
            final Response httpResponse = get(uri,
                    requestDescription.getHeaders(),
                    requestDescription.getParameters());

            return transformByteArrayToHttpResult(httpResponse, HttpUtils.SC_OK, convertFunction);
        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    /**
     * Select first or return absent.
     *
     * @param <T>                 the type parameter
     * @param listedRecordsResult the listed records result
     * @param predicate           the predicate
     * @return the http result
     */
    protected <T> HttpResult<T> selectFirstOrReturnAbsent(final HttpResult<List<T>> listedRecordsResult, final SDKPredicate<T> predicate) {
        final HttpResult.Builder<T> httpResultBuilder = new HttpResult.Builder<T>();
        httpResultBuilder.setHttpStatus(listedRecordsResult.getStatusCode());

        if (listedRecordsResult.isETagPresent()) {
            httpResultBuilder.setETagValue(listedRecordsResult.getETag().getValue());
        }

        if (!listedRecordsResult.isPresent()) {
            return httpResultBuilder.build();
        }

        final List<T> records = listedRecordsResult.get();

        for (final T record : records) {
            if (predicate.test(record)) {
                return httpResultBuilder.setHttpStatus(HttpUtils.SC_OK)
                        .setValue(record)
                        .build();
            }
        }

        return httpResultBuilder.build();
    }

    /**
     * Gets and transform to list.
     *
     * @param uri                the uri
     * @param requestDescription the request description
     * @param jsonEntry          the json entry
     * @param function           the function
     * @return the and transform to list
     * @throws ClientException the client exception
     */
    protected <T> HttpResult<List<T>> getAndTransformToList(final String uri,
                                                            final RequestDescription requestDescription,
                                                            final String jsonEntry,
                                                            final SDKFunction<JSONObject, T> function) throws ClientException {
        Response httpResponse = null;
        try {
            httpResponse = get(uri,
                    requestDescription.getHeaders(),
                    requestDescription.getParameters());
            return transformJSONToList(httpResponse,
                    HttpUtils.SC_OK,
                    jsonEntry,
                    function);
        } catch (Exception e) {
            Logger.e(TAG + " Error ocures when get called", e);
            throw new ClientException(e);
        } finally {
            closeSilently(httpResponse);
        }
    }


    protected <T> HttpResult<List<T>> getAndTransformToList(final String uri,
                                                            final RequestDescription requestDescription,
                                                            final int[] expectedStatuses,
                                                            final String jsonEntry,
                                                            final SDKFunction<JSONObject, T> function) throws ClientException {
        Response response = null;
        try {
            response = get(uri,
                    requestDescription.getHeaders(),
                    requestDescription.getParameters());
            return transformJSONToList(response,
                    expectedStatuses,
                    jsonEntry,
                    function);
        } catch (Exception e) {
            throw new ClientException(e);
        } finally {
            closeSilently(response);
        }
    }

    protected <T> HttpResult<List<T>> postAndTransformToList(final String uri,
                                                             final RequestDescription requestDescription,
                                                             final int[] expectedStatuses,
                                                             final String jsonEntry,
                                                             final SDKFunction<JSONObject, T> function) throws ClientException {
        Response response = null;
        try {
            response = post(uri,
                    requestDescription.getHeaders(),
                    requestDescription.getParameters());
            return transformJSONToList(response,
                    expectedStatuses,
                    jsonEntry,
                    function);
        } catch (Exception e) {
            throw new ClientException(e);
        } finally {
            closeSilently(response);
        }
    }


    /**
     * Create and transform.
     *
     * @param <T>             the type parameter
     * @param uri             the uri
     * @param entityData      the entity data
     * @param convertFunction the convert function
     * @return the http result
     * @throws ClientException the client exception
     */
    protected <T> HttpResult<T> createAndTransform(final String uri,
                                                   final AbstractEntityData entityData,
                                                   final SDKFunction<JSONObject, T> convertFunction) throws ClientException {
        Response httpResponse = null;
        try {
            httpResponse = post(uri, entityData.getParameters());

            return transformJSONToEntity(httpResponse, HttpUtils.SC_CREATED, convertFunction);
        } catch (Exception e) {
            throw new ClientException(e);
        } finally {
            closeSilently(httpResponse);
        }
    }

    /**
     * Create and transform convenience method.
     *
     * @param <T>                the type parameter
     * @param uri                the uri
     * @param requestDescription the request description
     * @param convertFunction    the convert function
     * @return the http result
     * @throws ClientException the client exception
     */
    protected <T> HttpResult<T> createAndTransform(final String uri,
                                                   final RequestDescription requestDescription,
                                                   final SDKFunction<JSONObject, T> convertFunction) throws ClientException {
        Response httpResponse = null;
        try {
            httpResponse = post(uri, requestDescription.getHeaders(), requestDescription.getParameters());
            return transformJSONToEntity(httpResponse, HttpUtils.SC_CREATED, convertFunction);
        } catch (Exception e) {
            throw new ClientException(e);
        } finally {
            closeSilently(httpResponse);
        }
    }

    /**
     * Create and transform convenience method.
     *
     * @param <K>             the type parameter
     * @param <T>             the type parameter
     * @param uri             the uri
     * @param entityData      the entity data
     * @param convertFunction the convert function
     * @param buildFunction   the build function
     * @return the http result
     * @throws ClientException the client exception
     */
    protected <K, T> HttpResult<T> createAndTransform(final String uri,
                                                      final AbstractEntityData entityData,
                                                      final SDKThrowableFunction<ResponseBody, K, Exception> convertFunction,
                                                      final SDKFunction<K, T> buildFunction) throws ClientException {
        Response response = null;
        try {
            response = post(uri, entityData.getParameters());
            return transform(response, new int[]{HttpUtils.SC_CREATED}, convertFunction, buildFunction);
        } catch (Exception e) {
            throw new ClientException(e);
        } finally {
            closeSilently(response);
        }
    }

    /**
     * Gets and transform to list convenience method.
     *
     * @param uri                the uri
     * @param requestDescription the request description
     * @param convertFunction    the convert function
     * @param buildFunction      the build function
     * @return the and transform to list
     * @throws ClientException the client exception
     */
    protected <K, T> HttpResult<T> getAndTransformToList(final String uri,
                                                         final RequestDescription requestDescription,
                                                         final SDKThrowableFunction<ResponseBody, K, Exception> convertFunction,
                                                         final SDKFunction<K, T> buildFunction) throws ClientException {
        Response response = null;
        try {
            response = get(uri, requestDescription.getHeaders(), requestDescription.getParameters());
            return transform(response, new int[]{HttpUtils.SC_OK}, convertFunction, buildFunction);
        } catch (Exception e) {
            throw new ClientException(e);
        } finally {
            closeSilently(response);
        }
    }

    /**
     * Post and return http status convenience method.
     *
     * @param uri                the uri
     * @param requestDescription the request description
     * @return the int
     * @throws ClientException the client exception
     */
    protected int postAndReturnHttpStatus(String uri, RequestDescription requestDescription) throws ClientException {
        Response httpResponse = null;
        try {
            httpResponse = post(uri,
                    requestDescription.getHeaders(),
                    requestDescription.getParameters());
            return httpStatusCode(httpResponse);
        } catch (Exception e) {
            throw new ClientException(e);
        } finally {
            closeSilently(httpResponse);
        }
    }

    private Request buildPostRequest(final String uri, final Headers headers, List<Map.Entry<String, String>> parameters) {
        Headers.Builder headersBuilder = standardHeaders.newBuilder();

        for (String headerName : headers.names()) {
            headersBuilder.add(headerName, headers.get(headerName));
        }

        Headers requestHeaders = headersBuilder.build();

        final Request.Builder requestBuilder = new Request.Builder()
                .post(convertToRequestBody(parameters))
                .url(baseUri.buildUpon().appendEncodedPath(URLEncoder.encodeUrl(uri)).build().toString())
                .headers(requestHeaders);

        for (final String headerName : headers.names()) {
            requestBuilder.addHeader(headerName, headers.get(headerName));
        }

        return requestBuilder.build();
    }

    private Request buildPostRequest(final String uri, final Headers headers, final JSONObject payload) {
        Headers.Builder headersBuilder = standardHeaders.newBuilder();

        for (String headerName : headers.names()) {
            headersBuilder.add(headerName, headers.get(headerName));
        }

        Headers requestHeaders = headersBuilder.build();
        RequestBody body = RequestBody.create(MEDIA_TYPE_JSON, payload.toString());

        final Request.Builder requestBuilder = new Request.Builder()
                .post(body)
                .url(baseUri.buildUpon()
                        .appendEncodedPath(URLEncoder.encodeUrl(uri)).build().toString())
                .headers(requestHeaders);

        return requestBuilder.build();
    }

    private Request buildGetRequest(final String uri,
                                    final Headers headers,
                                    final List<Map.Entry<String, String>> parameters) {
        Uri.Builder uriBuilder = baseUri.buildUpon().appendEncodedPath(URLEncoder.encodeUrl(uri));

        for (Map.Entry<String, String> parameterEntry : parameters) {
            uriBuilder.appendQueryParameter(parameterEntry.getKey(), parameterEntry.getValue());
        }

        Headers.Builder headersBuilder = standardHeaders.newBuilder();

        for (String headerName : headers.names()) {
            headersBuilder.add(headerName, headers.get(headerName));
        }

        Headers requestHeaders = headersBuilder.build();

        return new Request.Builder()
                .url(uriBuilder.build().toString())
                .headers(requestHeaders)
                .build();
    }

    private void closeSilently(Response httpResponse) {
        if (httpResponse != null) {
            try {
                httpResponse.body().close();
            } catch (IOException ingored) {
            }
        }
    }

    private static RequestBody convertToRequestBody(final List<Map.Entry<String, String>> parameterList) {

        if (parameterList.isEmpty()) {
            return RequestBody.create(MEDIA_TYPE_APPLICATION_FORM_URLENCODED, "");
        }

        FormEncodingBuilder requestBodyBuilder = new FormEncodingBuilder();
        for (Map.Entry<String, String> entry : parameterList) {
            String value = entry.getValue();
            if (value != null) {
                requestBodyBuilder.add(entry.getKey(), value);
            }
        }

        return requestBodyBuilder.build();
    }

    static RequestDescription returnDescriptionWithEtagOrDefault(final SDKOptional<ETag> etag) {
        return etag.isPresent() ? RequestDescription.start()
                .setETag(etag.get())
                .build()
                :
                DEFAULT_REQUEST_DESCRIPTION;
    }

    static Headers returnEmptyOrRequestETagHeader(final SDKOptional<ETag> eTagOptional) {
        Headers.Builder builder = new Headers.Builder();

        if (eTagOptional.isPresent()) {
            ETag object = eTagOptional.get();
            builder.add(object.getRequestName(), object.getValue());
        }

        return builder.build();
    }
}
