package com.kontakt.sdk.android.common.util;

import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.common.log.Logger;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.UUID;

/**
 * Converter provides utility methods which useful when working with Beacons.
 */
public final class ConversionUtils {

  private ConversionUtils() {
  }

  /**
   * Converts byte to integer value.
   *
   * @param value the value
   * @return the int
   */
  public static int asInt(final byte value) {
    return value & 0xFF;
  }

  public static byte[] extractSubdata(final byte[] packet, final int start, final int length) {
    SDKPreconditions.checkNotNull(packet, "Source array is null");

    if (packet.length < (start + length)) {
      Logger.e("Cannot extractBytes payload. Source array is too short: " + Arrays.toString(packet));
      return null;
    }

    final byte[] array = new byte[length];

    System.arraycopy(packet, start, array, 0, length);

    return array;
  }

  /**
   * checkes whether array starts with the byte sequence specified in prefix array.
   *
   * @param array the array
   * @param prefix the prefix array
   * @return returns true if array starts with bytes sequence specified in prefix array
   */
  public static boolean doesArrayBeginWith(final byte[] array, final byte[] prefix) {
    if (array.length < prefix.length) {
      return false;
    }

    for (int i = 0, size = prefix.length; i < size; i++) {
      if (array[i] != prefix[i]) {
        return false;
      }
    }

    return true;
  }

  /**
   * Inverts bytes array.
   *
   * @param array the array
   * @return the byte [ ]
   */
  public static byte[] invert(final byte[] array) {
    final int size = array.length;
    final int limit = size / 2;
    int i = 0;

    while (i < limit) {
      byte temp = array[i];
      array[i] = array[size - 1 - i];
      array[size - 1 - i] = temp;
      i++;
    }

    return array;
  }

  /**
   * Converts byte array with little endian order into integer value.
   *
   * @param input the input array
   * @return the int value
   */
  public static int asIntFromLittleEndianBytes(byte[] input) {
    SDKPreconditions.checkArgument(input != null && input.length > 0, "Input byte array is null or empty.");
    return asInt(invert(input));
  }

  public static float asFloatFromLittleEndianBytes(byte[] input) {
    SDKPreconditions.checkArgument(input != null && input.length > 0, "Input byte array is null or empty.");
    return ByteBuffer.wrap(invert(input)).getFloat();
  }

  /**
   * Converts array into integer value.
   *
   * @param input the input array
   * @return the int value
   */
  public static int asInt(byte[] input) {
    SDKPreconditions.checkArgument(input != null && input.length > 0, "Input byte array is null or empty.");

    final byte[] result = new byte[4];
    result[0] = 0;
    result[1] = 0;
    result[2] = 0;
    result[3] = 0;

    switch (input.length) {

      case 1:
        return asInt(input[0]);
      case 2:
        result[2] = input[0];
        result[3] = input[1];
        break;
      case 3:
        result[1] = input[0];
        result[2] = input[1];
        result[3] = input[2];
        break;
      case 4:
        result[0] = input[0];
        result[1] = input[1];
        result[2] = input[2];
        result[3] = input[3];
        break;
      default:
        throw new IllegalArgumentException("Input byte array exceeds max integer size (4 bytes)");
    }

    return ByteBuffer.wrap(result).getInt();
  }

  /**
   * Converts integer value to 2-byte array.
   *
   * @param value the value
   * @return the byte [ ]
   */
  public static byte[] to2ByteArray(final int value) {
    return new byte[] { (byte) value, (byte) (value >> 8) };
  }

  /**
   * Converts UUID ({@link UUID}) to bytes array.
   *
   * @param uuid the uuid
   * @return the byte array
   */
  public static byte[] convert(final UUID uuid) {
    final long msb = uuid.getMostSignificantBits();
    final long lsb = uuid.getLeastSignificantBits();
    return ByteBuffer.allocate(16).putLong(msb).putLong(lsb).array();
  }

  /**
   * Converts File to bytes array.
   *
   * @param file the file
   * @return the byte [ ]
   * @throws IOException the iO exception
   */
  public static byte[] convert(final File file) throws IOException {
    SDKPreconditions.checkArgument(file.length() < Integer.MAX_VALUE, "File size is too big.");

    byte[] buffer = new byte[(int) file.length()];
    InputStream stream = null;
    try {
      stream = new FileInputStream(file);
      if (stream.read(buffer) == -1) {
        throw new IOException("EOF reached while trying to read the whole file");
      }
    } finally {
      Closeables.closeQuietly(stream);
    }

    return buffer;
  }

  /**
   * Converts bytes array to Beacon power level.
   *
   * @param byteValue the byte value
   * @return the int
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  public static int toPowerLevel(final byte[] byteValue) {
    SDKPreconditions.checkArgument(byteValue.length == 1, "Specified value should be 1 byte long.");
    return toPowerLevel(byteValue[0]);
  }

  /**
   * Converts integer value to power level. The value should be obtained from Beacon device.
   *
   * @param value the value
   * @return the int
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  public static int toPowerLevel(final int value) {
    switch (value) {

      case -30:
        return 0;
      case -20:
        return 1;
      case -16:
        return 2;
      case -12:
        return 3;
      case -8:
        return 4;
      case -4:
        return 5;
      case 0:
        return 6;
      case 4:
        return 7;
      default:
        throw new IllegalArgumentException("Unsupported power level value.");
    }
  }

  /**
   * Converts power level to bytes array which can be accepted by Beacon.
   *
   * @param powerLevel the power level
   * @return the byte [ ] array
   */
  public static byte[] convertPowerLevel(final int powerLevel) {
    switch (powerLevel) {
      case 0:
        return new byte[] { (byte) 0xe2 };
      case 1:
        return new byte[] { (byte) 0xec };
      case 2:
        return new byte[] { (byte) 0xf0 };
      case 3:
        return new byte[] { (byte) 0xf4 };
      case 4:
        return new byte[] { (byte) 0xf8 };
      case 5:
        return new byte[] { (byte) 0xfc };
      case 6:
        return new byte[] { 0x00 };
      case 7:
        return new byte[] { 0x04 };
      default:
        throw new IllegalArgumentException(String.format("Unsupported power level: %d", powerLevel));
    }
  }

  /**
   * Converts TX Power level to its decimal dBm representation.
   */
  public static int fromPowerLevelToDbm(int txPower) {
    SDKPreconditions.checkArgument(txPower >= 0 && txPower <= 7, "Tx Power should be between 0 and 7");
    switch (txPower) {
      case 0:
        return -30;
      case 1:
        return -20;
      case 2:
        return -16;
      case 3:
        return -12;
      case 4:
        return -8;
      case 5:
        return -4;
      case 6:
        return 0;
      case 7:
        return 4;
    }
    return -1;
  }

  /**
   * Converts bytes array to UUID
   *
   * @param uuid the byte array
   * @return the uUID
   */
  public static UUID toUUID(final byte[] uuid) {
    final ByteBuffer byteBuffer = ByteBuffer.wrap(uuid);
    return new UUID(byteBuffer.getLong(), byteBuffer.getLong());
  }

  public static boolean doesArrayContainSubset(byte[] src, byte[] subset, int startIndex) {
    if (src == null || subset == null) {
      return false;
    }

    if ((subset.length + startIndex) > src.length) {
      return false;
    }

    if (startIndex < 0) {
      return false;
    }

    for (int i = startIndex, size = startIndex + subset.length; i < size; i++) {
      if (src[i] != subset[i - startIndex]) {
        return false;
      }
    }

    return true;
  }

  public static boolean containsScanResponse(byte[] scanRecord) {
    return !(scanRecord == null) && (ConversionUtils.contains(scanRecord, ScanResponse.TYPE_SERVICE_DATA) && ConversionUtils.contains(scanRecord,
        ScanResponse.D00D_SERVICE_UUID_LSB) && ConversionUtils.contains(scanRecord, ScanResponse.D00D_SERVICE_UUID_MSB));
  }

  public static boolean contains(byte[] source, int valueToFind) {
    for (byte b : source) {
      if (b == valueToFind) {
        return true;
      }
    }
    return false;
  }

  /**
   * Converts HEX string to byte array.
   *
   * @param hexString input HEX string
   * @return byte array
   */
  public static byte[] hexStringToByteArray(String hexString) {
    int len = hexString.length();
    byte[] data = new byte[len / 2];
    for (int i = 0; i < len - 1; i += 2) {
      data[i / 2] = (byte) ((Character.digit(hexString.charAt(i), 16) << 4) + Character.digit(hexString.charAt(i + 1), 16));
    }
    return data;
  }
}
