package com.kontakt.sdk.android.cloud.api.executor.places;

import android.util.Base64;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.PlacesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.PlacesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Place;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

/**
 * Request executor provided by {@link PlacesApi}. Use this class if you want to create
 * places through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.places().create(place)
 *      .withSchemaFile(new File(fileName))
 *      .execute();
 *   </code>
 * </pre>
 */
public class CreatePlaceRequestExecutor extends RequestExecutor<Place> {

  private final PlacesService placesService;
  private final Place place;
  private String encodedSchemaFile;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param placesService the places API service.
   * @param place the place we want to create.
   */
  public CreatePlaceRequestExecutor(PlacesService placesService, Place place) {
    this.placesService = placesService;
    this.place = place;
  }

  /**
   * Specifies a schema file of newly created place.
   *
   * @param schemaFile the schema file.
   * @return this request executor.
   * @throws IOException instance of IOException.
   */
  public CreatePlaceRequestExecutor withSchemaFile(final File schemaFile) throws IOException {
    checkNotNull(schemaFile, "schema file is null");
    checkState(schemaFile.exists(), "schema file does not exist");
    final byte[] bytes = ConversionUtils.convert(schemaFile);
    this.encodedSchemaFile = Base64.encodeToString(bytes, Base64.DEFAULT);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Place execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Place> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Place> prepareCall() {
    return placesService.createPlace(params());
  }

  private void checkPreconditions() {
    checkState(place.getName() != null, "cannot create place - specify name");
    checkState(place.getCoordinates() != null || place.getGeoCoordinates() != null,
        "cannot create place - specify either coordinates or geo coordinates");
    checkState(!(place.getCoordinates() != null && place.getGeoCoordinates() != null),
        "cannot create place - both coordinates and geo coordinates cannot be specified");
    checkState(place.getParentId() != null || place.getVenueId() != null,
        "cannot create place - either parent ID or venue ID must be set");
    if (place.getVenueId() != null) {
      checkState(encodedSchemaFile != null, "cannot create place - schema file is required for root place");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Places.NAME_PARAMETER, place.getName());
    params.put(CloudConstants.Places.SCALE_PARAMETER, String.valueOf(place.getScale()));

    if (place.getCoordinates() != null) {
      params.put(CloudConstants.Places.COORDINATES_PARAMETER, place.getCoordinates().toString());
    }
    if (place.getGeoCoordinates() != null) {
      params.put(CloudConstants.Places.GEO_COORDINATES_PARAMETER, place.getGeoCoordinates().toString());
    }
    if (encodedSchemaFile != null) {
      params.put(CloudConstants.Places.SCHEMA_PARAMETER, encodedSchemaFile);
    }
    if (place.getParentId() != null) {
      params.put(CloudConstants.Places.PARENT_ID_PARAMETER, place.getParentId().toString());
    }
    if (place.getVenueId() != null) {
      params.put(CloudConstants.Places.VENUE_ID_PARAMETER, place.getVenueId().toString());
    }

    return params;
  }
}
