package com.kontakt.sdk.android.cloud.api.executor.configs;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ConfigsApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ConfigsService;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Message;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link ConfigsApi}. Use this class if you want to delete
 * pending configs through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.configs().delete(IDs).execute();
 *   </code>
 * </pre>
 */
public class DeletePendingConfigsRequestExecutor extends RequestExecutor<Message> {

  private final ConfigsService configsService;
  private final String[] uniqueIds;

  /**
   * Constructs request executor initialized with corresponding service class and config unique identifiers.
   *
   * @param configsService the configs API service.
   * @param uniqueIds pending config unique identifiers.
   */
  public DeletePendingConfigsRequestExecutor(final ConfigsService configsService, final String... uniqueIds) {
    this.configsService = configsService;
    this.uniqueIds = uniqueIds;
  }

  /**
   * Constructs request executor initialized with corresponding service class and config unique identifiers.
   *
   * @param configsService the configs API service.
   * @param uniqueIds pending config unique identifiers.
   */
  public DeletePendingConfigsRequestExecutor(final ConfigsService configsService, final List<String> uniqueIds) {
    this.configsService = configsService;
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Message> prepareCall() {
    return configsService.deletePendingConfigs(params());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    return params;
  }
}
