package com.kontakt.sdk.android.cloud.api.executor.configs;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ConfigsApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ConfigsService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.paginated.Configs;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link ConfigsApi}. Use this class if you want to obtain
 * configs through fluent API in chained fashion. Here is an example of how to use this executor:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Configs configs = kontaktCloud.configs().fetch()
 *      .type(DeviceType.BEACON)
 *      .profile(DeviceProfile.EDDYSTONE)
 *      .maxResult(15)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that device type must be specified so invocation of {@code type} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class ConfigsRequestExecutor extends PaginatedRequestExecutor<Configs> {

  private final ConfigsService configsService;

  private DeviceType type;
  private DeviceProfile profile;
  private String[] uniqueIds;
  private boolean fetchPassword;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param configsService the configs API service.
   */
  public ConfigsRequestExecutor(final ConfigsService configsService) {
    this.configsService = configsService;
  }

  /**
   * Specifies device type. The method invocation is obligatory while using this request executor.
   *
   * @param type the device type.
   * @return this request executor.
   */
  public ConfigsRequestExecutor type(final DeviceType type) {
    this.type = SDKPreconditions.checkNotNull(type, "type cannot be null");
    return this;
  }

  /**
   * Specifies device profile.
   *
   * @param profile the device profile.
   * @return this request executor.
   */
  public ConfigsRequestExecutor profile(final DeviceProfile profile) {
    this.profile = SDKPreconditions.checkNotNull(profile, "profile cannot be null");
    return this;
  }

  /**
   * Specifies config unique identifiers. Use that method if you want to fetch particular configs by ID.
   *
   * @param uniqueIds config unique identifiers.
   * @return this request executor.
   */
  public ConfigsRequestExecutor withIds(final String... uniqueIds) {
    this.uniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies config unique identifiers. Use that method if you want to fetch particular configs by ID.
   *
   * @param uniqueIds config unique identifiers.
   * @return this request executor.
   */
  public ConfigsRequestExecutor withIds(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigsRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigsRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigsRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigsRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigsRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  public ConfigsRequestExecutor alsoFetchPassword(){
    this.fetchPassword = true;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Configs execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<Configs> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Configs> prepareCall() {
    final Call<Configs> call;
    if (eTag != null) {
      call = configsService.getConfigs(params(), eTag);
    } else {
      call = configsService.getConfigs(params());
    }
    return call;
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(type != null, "Cannot get configs - specify device type");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (type != null) {
      params.put(CloudConstants.Configs.DEVICE_TYPE_PARAMETER, type.name());
    }
    if (profile != null) {
      params.put(CloudConstants.Configs.PROFILE_PARAMETER, profile.name());
    }
    if (uniqueIds != null && uniqueIds.length > 0) {
      params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    }
    if(fetchPassword && type != null){
      params.put("clientType", type.name());
    }
    return params;
  }
}
