package com.kontakt.sdk.android.cloud.api.executor;

import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.util.HttpCodes;
import java.io.IOException;
import java.util.Map;
import retrofit2.Call;
import retrofit2.Response;

/**
 * Base request executor provides methods to execute customized requests both in sync and async mode.
 *
 * @param <T> the response type
 */
public abstract class RequestExecutor<T> {

  /**
   * Executes created request synchronously.
   *
   * @return the response object.
   * @throws IOException exception
   * @throws KontaktCloudException exception
   */
  public T execute() throws IOException, KontaktCloudException {
    final Call<T> call = prepareCall();
    final Response<T> response = call.execute();
    if (response.isSuccessful() || response.code() == HttpCodes.SC_NOT_MODIFIED) {
      return response.body();
    }
    String rawMessage = response.raw().message();
    if(response.errorBody() != null){
      rawMessage += "; " + response.errorBody().string();
    }
    final int code = response.code();
    final String message = "code - " + code + ", message - " + rawMessage;
    throw new KontaktCloudException(message, code);
  }

  /**
   * Executes created request asynchronously and invokes callback with the result.
   *
   * @param callback the callback object.
   */
  public void execute(final CloudCallback<T> callback) {
    final Call<T> call = prepareCall();
    call.enqueue(new CloudCallbackWrapper<>(callback));
  }

  /**
   * Prepares a request to execute.
   *
   * @return the request.
   */
  protected abstract Call<T> prepareCall();

  /**
   * Composes the request's parameters in map.
   *
   * @return the map of request body parameters that are being used to update data on server.
   */
  protected abstract Map<String, String> params();
}
