package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.triggers.CreateTriggerRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.triggers.DeleteTriggerRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.triggers.TriggersRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.triggers.UpdateTriggerRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.TriggersService;
import com.kontakt.sdk.android.common.model.Trigger;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * An abstract representation of Triggers API endpoint. Provides executors that can be
 * used to handle all trigger-related requests.
 */
public class TriggersApi {

  private final TriggersService triggersService;

  /**
   * Constructs Triggers API endpoint initialized with corresponding service class.
   *
   * @param triggersService the triggers API service.
   */
  public TriggersApi(final TriggersService triggersService) {
    this.triggersService = triggersService;
  }

  /**
   * Provides executor that can be used to obtain triggers.
   *
   * @return the request executor.
   */
  public TriggersRequestExecutor fetch() {
    return new TriggersRequestExecutor(triggersService);
  }

  /**
   * Provides executor that can be used to create new trigger.
   *
   * @param trigger the trigger instance.
   * @return the request executor.
   */
  public CreateTriggerRequestExecutor create(final Trigger trigger) {
    checkNotNull(trigger, "Trigger cannot be null");
    return new CreateTriggerRequestExecutor(triggersService, trigger);
  }

  /**
   * Provides executor that can be used to update a venue.
   *
   * @param id the venue's unique identifier.
   * @return the request executor.
   */
  public UpdateTriggerRequestExecutor update(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new UpdateTriggerRequestExecutor(triggersService, id);
  }

  /**
   * Provides executor that can be used to delete a trigger.
   *
   * @param id the trigger's unique identifier.
   * @return the request executor.
   */
  public DeleteTriggerRequestExecutor delete(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new DeleteTriggerRequestExecutor(triggersService, id);
  }
}
