package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.devices.ApplySecureConfigRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.AssignDeviceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.CredentialsListRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.CredentialsRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.DevicesFromUrlRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.DevicesRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.EddystonesRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.IBeaconsRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.MoveDeviceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.RegisterDeviceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.ShareDeviceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.TelemetryRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.UnassignDeviceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.UnshareDeviceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.devices.UpdateDeviceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.UUID;

/**
 * An abstract representation of Devices API endpoint. Provides executors that can be
 * used to handle all device-related requests.
 */
public class DevicesApi {

  private final DevicesService devicesService;

  /**
   * Constructs Devices API endpoint initialized with corresponding service class.
   *
   * @param devicesService the actions API service.
   */
  public DevicesApi(final DevicesService devicesService) {
    this.devicesService = devicesService;
  }

  /**
   * Provides executor that can be used to obtain devices.
   *
   * @return the request executor.
   */
  public DevicesRequestExecutor fetch() {
    return new DevicesRequestExecutor(devicesService);
  }

  /**
   * Provides executor responsible for loading devices from URL.
   *
   * @param url the URL.
   * @return the request executor.
   */
  public DevicesFromUrlRequestExecutor fetchFromUrl(final String url) {
    SDKPreconditions.checkNotNullOrEmpty(url, "url cannot be either null or empty");
    return new DevicesFromUrlRequestExecutor(devicesService, url);
  }

  /**
   * Provides executor that can be used to obtain Eddystone devices.
   *
   * @return the request executor.
   */
  public EddystonesRequestExecutor eddystones() {
    return new EddystonesRequestExecutor(devicesService);
  }

  /**
   * Provides executor that can be used to obtain iBeacon devices.
   *
   * @return the request executor.
   */
  public IBeaconsRequestExecutor ibeacons() {
    return new IBeaconsRequestExecutor(devicesService);
  }

  /**
   * Provides executor that can be used to obtain single device credentials.
   *
   * @param uniqueId the device's unique identifier.
   * @return the request executor.
   */
  public CredentialsRequestExecutor credentials(final String uniqueId) {
    SDKPreconditions.checkNotNull(uniqueId, "ID cannot be null");
    return new CredentialsRequestExecutor(devicesService, uniqueId);
  }

  /**
   * Provides executor that can be used to obtain multiple device credentials.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public CredentialsListRequestExecutor credentials(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new CredentialsListRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to decrypt encrypted TLM frame
   *
   * @param eid eddystone EID
   * @param etlmFrame encrypted TLM frame
   * @return the request executor.
   */
  public TelemetryRequestExecutor decryptTelemetry(final String eid, final String etlmFrame) {
    SDKPreconditions.checkNotNull(eid, "EID cannot be null");
    SDKPreconditions.checkNotNull(etlmFrame, "ETLM cannot be null");
    return new TelemetryRequestExecutor(devicesService, eid, etlmFrame);
  }

  /**
   * Provides executor that can be used to update a device.
   *
   * @param uniqueId the device's unique identifier.
   * @return the request executor.
   */
  public UpdateDeviceRequestExecutor update(final String uniqueId) {
    SDKPreconditions.checkNotNull(uniqueId, "ID cannot be null");
    return new UpdateDeviceRequestExecutor(devicesService, uniqueId);
  }

  /**
   * Provides executor that can be used to assign devices either to venue or to manager.
   *
   * @param deviceIds device unique identifiers.
   * @return the request executor.
   */
  public AssignDeviceRequestExecutor assign(final UUID... deviceIds) {
    SDKPreconditions.checkNotNull(deviceIds, "IDs cannot be null");
    return new AssignDeviceRequestExecutor(devicesService, deviceIds);
  }

  /**
   * Provides executor that can be used to assign devices either to venue or to manager.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public AssignDeviceRequestExecutor assign(final String... uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new AssignDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to unassign devices from venue.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public UnassignDeviceRequestExecutor unassignFromVenue(final String... uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new UnassignDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to unassign devices from venue.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public UnassignDeviceRequestExecutor unassignFromVenue(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new UnassignDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to move devices to another manager or company.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public MoveDeviceRequestExecutor move(final String... uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new MoveDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to move devices to another manager or company.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public MoveDeviceRequestExecutor move(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new MoveDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to share devices with other managers.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public ShareDeviceRequestExecutor share(final String... uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new ShareDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to share devices with other managers.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public ShareDeviceRequestExecutor share(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new ShareDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to unshare devices from managers.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public UnshareDeviceRequestExecutor unshare(final String... uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new UnshareDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to unshare devices from managers.
   *
   * @param uniqueIds device unique identifiers.
   * @return the request executor.
   */
  public UnshareDeviceRequestExecutor unshare(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new UnshareDeviceRequestExecutor(devicesService, uniqueIds);
  }

  /**
   * Provides executor that can be used to apply secure configs on devices.
   *
   * @param configs secure configs.
   * @return the request executor.
   */
  public ApplySecureConfigRequestExecutor applySecureConfigs(final Config... configs) {
    SDKPreconditions.checkNotNull(configs, "configs cannot be null");
    return new ApplySecureConfigRequestExecutor(devicesService, configs);
  }

  /**
   * Provides executor that can be used to apply secure configs on devices.
   *
   * @param configs secure configs.
   * @return the request executor.
   */
  public ApplySecureConfigRequestExecutor applySecureConfigs(final List<Config> configs) {
    SDKPreconditions.checkNotNull(configs, "configs cannot be null");
    return new ApplySecureConfigRequestExecutor(devicesService, configs);
  }

  /**
   * Provides executor that can be used to register the device.
   *
   * @param sourceId device's ID.
   * @return the request executor.
   */
  public RegisterDeviceRequestExecutor register(final String sourceId) {
    SDKPreconditions.checkNotNull(sourceId);
    return new RegisterDeviceRequestExecutor(devicesService, sourceId);
  }
}
