package com.kontakt.sdk.android.ble.spec;

import android.os.Parcel;
import android.os.Parcelable;

import static com.kontakt.sdk.android.common.util.ArrayUtils.binaryStringOfByte;

/**
 * Telemetry class that provides information about system health for battery operated devices, aggregates accelerometer data,
 * sensors and scanning.
 *
 * Scans per second encoding:
 * 0      |   Exactly zero scans over last 10 seconds
 * 0-249  |   Rounded number of scans/s.
 * 250    |   250 or more scans/s
 * 251    |   0.1 - 0.2 scans/s
 * 252    |   0.3 - 0.4 scans/s
 * 253    |   0.5 - 0.6 scans/s
 * 254    |   0.7 - 0.8 scans/s
 */
public class KontaktTelemetry implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Parcelable.Creator<KontaktTelemetry> CREATOR = new Parcelable.Creator<KontaktTelemetry>() {
    @Override
    public KontaktTelemetry createFromParcel(Parcel source) {
      return new KontaktTelemetry(source);
    }

    @Override
    public KontaktTelemetry[] newArray(int size) {
      return new KontaktTelemetry[size];
    }
  };

  public static final int BATTERY_LEVEL_EXTERNAL_POWER_SUPPLIED_VALUE = 255;
  public static final int DOUBLE_TAP_DISABLED_VALUE = 65535;
  public static final int THRESHOLD_DISABLED_VALUE = 65535;
  public static final int LOAD_AVERAGE_NOT_SUPPORTED_VALUE = 255;
  public static final int BLE_SCANS_NOT_SUPPORTED_VALUE = 255;
  public static final int WIFI_SCANS_NOT_SUPPORTED_VALUE = 255;
  public static final int BLE_DEVICES_NOT_SUPPORTED_VALUE = 65535;

  private final int timestamp;
  private final int batteryLevel;
  private final int uptime;
  private final int systemLoad;
  private final TelemetryError error;
  private final int sensitivity;
  private final Acceleration acceleration;
  private final int currentMovementId;
  private final int lastDoubleTap;
  private final int lastThreshold;
  private final int lightSensor;
  private final float temperature;
  private final int bleScans;
  private final int wifiScans;
  private final int bleDevices;
  private final int lastSingleClick;
  private final int singleClickCount;
  private final int lastDoubleClick;
  private final DeviceDataLoggerStatus dataLoggerStatus;
  private final int humidity;
  private final int bleChannel;
  private final int lastTap;
  private final byte definedGpioMask;
  private final byte stateGpioMask;
  private final float airPressure;
  private final int pirDetectionSeconds;
  private final int singleClickCount2;
  private final int manufacturerId;
  private final int airQuality;
  private final int roomNumber;
  private final int occupancy;
  private final int occupancyDetectionTime;

  private KontaktTelemetry(final Builder builder) {
    this.timestamp = builder.timestamp;
    this.batteryLevel = builder.batteryLevel;
    this.uptime = builder.uptime;
    this.systemLoad = builder.systemLoad;
    this.error = builder.error;
    this.sensitivity = builder.sensitivity;
    this.acceleration = builder.acceleration;
    this.lastDoubleTap = builder.lastDoubleTap;
    this.lastThreshold = builder.lastThreshold;
    this.lightSensor = builder.lightSensor;
    this.temperature = builder.temperature;
    this.bleScans = builder.bleScans;
    this.wifiScans = builder.wifiScans;
    this.bleDevices = builder.bleDevices;
    this.lastSingleClick = builder.lastSingleClick;
    this.singleClickCount = builder.singleClickCount;
    this.singleClickCount2 = builder.singleClickCount2;
    this.dataLoggerStatus = builder.dataLoggerStatus;
    this.humidity = builder.humidity;
    this.bleChannel = builder.bleChannel;
    this.lastTap = builder.lastTap;
    this.lastDoubleClick = builder.lastDoubleClick;
    this.currentMovementId = builder.currentMovementId;
    this.airPressure = builder.airPressure;
    this.definedGpioMask = builder.definedGpioMask;
    this.stateGpioMask = builder.stateGpioMask;
    this.pirDetectionSeconds = builder.pirDetectionSeconds;
    this.manufacturerId = builder.manufacturerId;
    this.airQuality = builder.airQuality;
    this.roomNumber = builder.roomNumber;
    this.occupancy = builder.occupancy;
    this.occupancyDetectionTime = builder.occupancyDetectionTime;
  }

  private KontaktTelemetry(Parcel in) {
    this.timestamp = in.readInt();
    this.batteryLevel = in.readInt();
    this.uptime = in.readInt();
    this.systemLoad = in.readInt();
    int tmpError = in.readInt();
    this.error = tmpError == -1 ? null : TelemetryError.values()[tmpError];
    this.sensitivity = in.readInt();
    this.acceleration = in.readParcelable(Acceleration.class.getClassLoader());
    this.lastDoubleTap = in.readInt();
    this.lastThreshold = in.readInt();
    this.lightSensor = in.readInt();
    this.temperature = in.readInt();
    this.bleScans = in.readInt();
    this.wifiScans = in.readInt();
    this.bleDevices = in.readInt();
    this.lastSingleClick = in.readInt();
    this.singleClickCount = in.readInt();
    this.dataLoggerStatus = DeviceDataLoggerStatus.valueOf(in.readString());
    this.humidity = in.readInt();
    this.bleChannel = in.readInt();
    this.lastTap = in.readInt();
    this.lastDoubleClick = in.readInt();
    this.currentMovementId = in.readInt();
    this.definedGpioMask = in.readByte();
    this.stateGpioMask = in.readByte();
    this.airPressure = in.readFloat();
    this.pirDetectionSeconds = in.readInt();
    this.singleClickCount2 = in.readInt();
    this.manufacturerId = in.readInt();
    this.airQuality = in.readInt();
    this.roomNumber = in.readInt();
    this.occupancy = in.readInt();
    this.occupancyDetectionTime = in.readInt();
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeInt(this.timestamp);
    dest.writeInt(this.batteryLevel);
    dest.writeInt(this.uptime);
    dest.writeInt(this.systemLoad);
    dest.writeInt(this.error == null ? -1 : this.error.ordinal());
    dest.writeInt(this.sensitivity);
    dest.writeParcelable(this.acceleration, flags);
    dest.writeInt(this.lastDoubleTap);
    dest.writeInt(this.lastThreshold);
    dest.writeInt(this.lightSensor);
    dest.writeFloat(this.temperature);
    dest.writeInt(this.bleScans);
    dest.writeInt(this.wifiScans);
    dest.writeInt(this.bleDevices);
    dest.writeInt(this.lastSingleClick);
    dest.writeInt(this.singleClickCount);
    dest.writeString(this.dataLoggerStatus.name());
    dest.writeInt(this.humidity);
    dest.writeInt(this.bleChannel);
    dest.writeInt(this.lastTap);
    dest.writeInt(this.lastDoubleClick);
    dest.writeInt(this.currentMovementId);
    dest.writeByte(this.definedGpioMask);
    dest.writeByte(this.stateGpioMask);
    dest.writeFloat(this.airPressure);
    dest.writeInt(this.pirDetectionSeconds);
    dest.writeInt(this.singleClickCount2);
    dest.writeInt(this.manufacturerId);
    dest.writeInt(this.airQuality);
    dest.writeInt(this.roomNumber);
    dest.writeInt(this.occupancy);
    dest.writeInt(this.occupancyDetectionTime);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    KontaktTelemetry that = (KontaktTelemetry) o;

    if (timestamp != that.timestamp) return false;
    if (batteryLevel != that.batteryLevel) return false;
    if (uptime != that.uptime) return false;
    if (systemLoad != that.systemLoad) return false;
    if (sensitivity != that.sensitivity) return false;
    if (lastDoubleTap != that.lastDoubleTap) return false;
    if (lastThreshold != that.lastThreshold) return false;
    if (lightSensor != that.lightSensor) return false;
    if (temperature != that.temperature) return false;
    if (bleScans != that.bleScans) return false;
    if (wifiScans != that.wifiScans) return false;
    if (bleDevices != that.bleDevices) return false;
    if (error != that.error) return false;
    if (lastSingleClick != that.lastSingleClick) return false;
    if (singleClickCount != that.singleClickCount) return false;
    if (dataLoggerStatus != that.dataLoggerStatus) return false;
    if (humidity != that.humidity) return false;
    if (bleChannel != that.bleChannel) return false;
    if(lastTap != that.lastTap) return false;
    if(lastDoubleClick != that.lastDoubleClick) return false;
    if(currentMovementId != that.currentMovementId) return false;
    if(airPressure != that.airPressure) return false;
    if(definedGpioMask != that.definedGpioMask) return false;
    if(stateGpioMask != that.stateGpioMask) return false;
    if(pirDetectionSeconds != that.pirDetectionSeconds) return false;
    if(singleClickCount2 != that.singleClickCount2) return false;
    if(manufacturerId != that.manufacturerId) return false;
    if(airQuality != that.airQuality) return false;
    if(roomNumber != that.roomNumber) return false;
    if(occupancy != that.roomNumber) return false;
    if(occupancyDetectionTime != that.occupancyDetectionTime) return false;
    //noinspection EqualsReplaceableByObjectsCall
    return acceleration != null ? acceleration.equals(that.acceleration) : that.acceleration == null;
  }


  @Override
  public int hashCode() {
    int result = timestamp;
    result = 31 * result + batteryLevel;
    result = 31 * result + uptime;
    result = 31 * result + systemLoad;
    result = 31 * result + (error != null ? error.hashCode() : 0);
    result = 31 * result + sensitivity;
    result = 31 * result + (acceleration != null ? acceleration.hashCode() : 0);
    result = 31 * result + lastDoubleTap;
    result = 31 * result + lastThreshold;
    result = 31 * result + lightSensor;
    result = 31 * result + Float.valueOf(temperature).hashCode();
    result = 31 * result + bleScans;
    result = 31 * result + wifiScans;
    result = 31 * result + bleDevices;
    result = 31 * result + lastSingleClick;
    result = 31 * result + singleClickCount;
    result = 31 * result + dataLoggerStatus.ordinal();
    result = 31 * result + humidity;
    result = 31 * result + bleChannel;
    result = 31 * result + lastTap;
    result = 31 * result + lastDoubleClick;
    result = 31 * result + singleClickCount;
    result = 31 * result + definedGpioMask;
    result = 31 * result + stateGpioMask;
    result = 31 * result + Float.valueOf(airPressure).hashCode();
    result = 31 * result + pirDetectionSeconds;
    result = 31 * result + singleClickCount2;
    result = 31 * result + manufacturerId;
    result = 31 * result + airQuality;
    result = 31 * result + roomNumber;
    result = 31 * result + occupancy;
    result = 31 * result + occupancyDetectionTime;
    return result;
  }

  @Override
  public String toString() {
    return "KontaktTelemetry{"
            + "timestamp="
            + timestamp
            + ", batteryLevel="
            + batteryLevel
            + ", uptime="
            + uptime
            + ", systemLoad="
            + systemLoad
            + ", error="
            + error
            + ", sensitivity="
            + sensitivity
            + ", acceleration="
            + acceleration
            + ", lastDoubleTap="
            + lastDoubleTap
            + ", lastThreshold="
            + lastThreshold
            + ", lightSensor="
            + lightSensor
            + ", temperature="
            + temperature
            + ", bleScans="
            + bleScans
            + ", wifiScans="
            + wifiScans
            + ", bleDevices="
            + bleDevices
            + ", lastSingleClick="
            + lastSingleClick
            + ", lastDoubleClick="
            + lastDoubleClick
            + ", lastTap="
            + lastTap
            + ", singleClickCount="
            + singleClickCount
            + ", singleClickCount2="
            + singleClickCount2
            + ", dataLoggerStatus="
            + dataLoggerStatus.name()
            + ", humidity="
            + humidity
            + ", bleChannel="
            + bleChannel
            + ",  airPressure="
            + airPressure
            + ", stateGpioMask: "
            + binaryStringOfByte(stateGpioMask)
            + ", definedGpioMask: "
            + binaryStringOfByte(definedGpioMask)
            + ", pirDetectionSeconds: "
            + pirDetectionSeconds
            + ", vendorSpecifics: "
            + "manufacturerId: "
            + manufacturerId
            + ", airQuality: "
            + airQuality
            + ", roomNumber: "
            + roomNumber
            + ", occupancy: "
            + occupancy
            + ", occupancy detection time: "
            + occupancyDetectionTime
            + '}';
  }



  @Override
  public int describeContents() {
    return 0;
  }


  /**
   * Returns Unix timestamp (seconds since 1970).
   *
   * @return the timestamp.
   */
  public int getTimestamp() {
    return timestamp;
  }

  /**
   * Returns battery level percentage (0 - 100).
   *
   * @return the battery level.
   */
  public int getBatteryLevel() {
    return batteryLevel;
  }

  /**
   * Returns uptime rounded to hours.
   *
   * @return the uptime.
   */
  public int getUptime() {
    return uptime;
  }

  /**
   * Returns percentage of system load average over 15 min.
   *
   * @return the system load percentage.
   */
  public int getSystemLoad() {
    return systemLoad;
  }

  /**
   * Returns error flags.
   *
   * @return the error.
   */
  public TelemetryError getError() {
    return error;
  }

  /**
   * Returns accelerometer sensitivity. Unit: mg/digit.
   *
   * @return the accelerometer sensitivity.
   */
  public int getSensitivity() {
    return sensitivity;
  }

  /**
   * Returns raw accelerometer data filtered with lowpass filter.
   *
   * @return the accelerometer data.
   */
  public Acceleration getAcceleration() {
    return acceleration;
  }

  /**
   * Returns seconds since last doubletap event. Doubletap event occurs when doubletap pattern is detected by accelerometer.
   * Saturates on 65535 (0xFFFF) value. 65535 (0xFFFF) is default value when Doubletap is disabled.
   *
   * @return the seconds since last doubletap event.
   */
  public int getLastDoubleTap() {
    return lastDoubleTap;
  }

  /**
   * Returns seconds since last threshold event. Threshold event occurs when acceleration exceeds configured threshold.
   * Saturates on 65535 (0xFFFF) value. 65535 (0xFFFF) is default value when Threshold is disabled.
   *
   * @return the seconds since last threshold event.
   */
  public int getLastThreshold() {
    return lastThreshold;
  }

  /**
   * Returns light sensor percentage (0 - 100).
   *
   * @return the light sensor percentage.
   */
  public int getLightSensor() {
    return lightSensor;
  }

  /**
   * Returns temperature in Celsius degrees.
   *
   * @return the temperature value.
   */
  public float getTemperature() {
    return temperature;
  }

  /**
   * Returns BLE scanning statistics (10 seconds average).
   *
   * @return the BLE scanning statistics.
   */
  public int getBleScans() {
    return bleScans;
  }

  /**
   * Returns WiFi scanning statistics (10 seconds average).
   *
   * @return the WiFi scanning statistics.
   */
  public int getWifiScans() {
    return wifiScans;
  }

  /**
   * Returns number of scanned BLE devices (10 seconds average).
   *
   * @return the scanned BLE devices amount.
   */
  public int getBleDevices() {
    return bleDevices;
  }

  /**
   * Returns seconds since last click event. Click event is specified in Button Specification Behavior.
   * Saturates on 65535 (0xFFFF) value. 65535 (0xFFFF) default value when button click is disabled.
   *
   * @return seconds since last click.
   */
  public int getLastSingleClick() {
    return lastSingleClick;
  }

  /**
   * Returns a number of click events. Click event is specified in Button Specification Behavior.
   *
   * @return the number of click events.
   */
  public int getSingleClickCount() {
    return singleClickCount;
  }

  /**
   * A value indicating whether the data logger functionality is turned `ON`.
   *
   * @return data logger enumeration status.
   */
  public DeviceDataLoggerStatus getDataLoggerStatus() {
    return dataLoggerStatus;
  }

  /**
   * Returns Relative Humidity Percentage (0-100).
   *
   * @return the humidity percentage value.
   */
  public int getHumidity() {
    return humidity;
  }

  /**
   * Returns BLE Channel value.
   *
   * @return the BLE channel value.
   */
  public int getBleChannel() {
    return bleChannel;
  }

  public int getLastTap() {
    return lastTap;
  }

  public int getLastDoubleClick() {
    return lastDoubleClick;
  }

  public int getCurrentMovementId() {
    return currentMovementId;
  }

  public float getAirPressure() {
    return airPressure;
  }

  public byte getDefinedGpioMask() {
    return definedGpioMask;
  }

  public byte getStateGpioMask() {
    return stateGpioMask;
  }

  public int getPirDetectionSeconds() {
    return pirDetectionSeconds;
  }

  public int getSingleClickCount2() {
    return singleClickCount2;
  }

  public int getAirQuality() {
    return airQuality;
  }

  public int getManufacturerId() {
    return manufacturerId;
  }

  public int getOccupancy() {
    return occupancy;
  }

  public int getOccupancyDetectionTime() {
    return occupancyDetectionTime;
  }

  public int getRoomNumber() {
    return roomNumber;
  }

  public static class Builder {

    private int timestamp = -1;
    private int batteryLevel = -1;
    private int uptime = -1;
    private int systemLoad = -1;
    private TelemetryError error = null;
    private int sensitivity = -1;
    private Acceleration acceleration = null;
    private int lastDoubleTap = -1;
    private int currentMovementId = -1;
    private int lastThreshold = -1;
    private int lightSensor = -1;
    private float temperature = -1;
    private int bleScans = -1;
    private int wifiScans = -1;
    private int bleDevices = -1;
    private int lastSingleClick = -1;
    private int singleClickCount = -1;
    private int lastDoubleClick = -1;
    private DeviceDataLoggerStatus dataLoggerStatus = DeviceDataLoggerStatus.UNAVAILABLE;
    private int humidity = -1;
    private int bleChannel = -1;
    private int lastTap = -1;
    private float airPressure = -1;
    private byte definedGpioMask = -1;
    private byte stateGpioMask = -1;
    private int pirDetectionSeconds = -1;
    private int singleClickCount2 = -1;
    private int manufacturerId = -1;
    private int airQuality = -1;
    private int roomNumber = -1;
    private int occupancy = -1;
    private int occupancyDetectionTime = -1;

    public Builder lastTap(int lastTap){
      this.lastTap = lastTap;
      return this;
    }

    public Builder timestamp(final int timestamp) {
      this.timestamp = timestamp;
      return this;
    }

    public Builder batteryLevel(final int batteryLevel) {
      this.batteryLevel = batteryLevel;
      return this;
    }

    public Builder uptime(final int uptime) {
      this.uptime = uptime;
      return this;
    }

    public Builder systemLoad(final int systemLoad) {
      this.systemLoad = systemLoad;
      return this;
    }

    public Builder error(final TelemetryError error) {
      this.error = error;
      return this;
    }

    public Builder sensitivity(final int sensitivity) {
      this.sensitivity = sensitivity;
      return this;
    }

    public Builder acceleration(final Acceleration acceleration) {
      this.acceleration = acceleration;
      return this;
    }

    public Builder lastDoubleTap(final int lastDoubleTap) {
      this.lastDoubleTap = lastDoubleTap;
      return this;
    }

    public Builder lastThreshold(final int lastThreshold) {
      this.lastThreshold = lastThreshold;
      return this;
    }

    public Builder lightSensor(final int lightSensor) {
      this.lightSensor = lightSensor;
      return this;
    }

    public Builder temperature(final float temperature) {
      this.temperature = temperature;
      return this;
    }

    public Builder bleScans(final int bleScans) {
      this.bleScans = bleScans;
      return this;
    }

    public Builder wifiScans(final int wifiScans) {
      this.wifiScans = wifiScans;
      return this;
    }

    public Builder bleDevices(final int bleDevices) {
      this.bleDevices = bleDevices;
      return this;
    }

    public Builder lastSingleClick(final int lastSingleClick) {
      this.lastSingleClick = lastSingleClick;
      return this;
    }

    public Builder singleClickCount(final int singleClickCount) {
      this.singleClickCount = singleClickCount;
      return this;
    }

    public Builder lastDoubleClick(int lastDoubleClick){
      this.lastDoubleClick = lastDoubleClick;
      return this;
    }

    public Builder dataLoggerStatus(final DeviceDataLoggerStatus dataLoggerStatus) {
      if (dataLoggerStatus != null) {
        this.dataLoggerStatus = dataLoggerStatus;
      }
      return this;
    }

    public Builder humidity(final int humidity) {
      this.humidity = humidity;
      return this;
    }

    public Builder bleChannel(final int bleChannel) {
      this.bleChannel = bleChannel;
      return this;
    }

    public Builder currentMovementId(int currentMovementId){
      this.currentMovementId = currentMovementId;
      return this;
    }

    public Builder definedGpioMask(byte definedGpioMask){
      this.definedGpioMask = definedGpioMask;
      return this;
    }

    public Builder stateGpioMask(byte stateGpioMask){
      this.stateGpioMask = stateGpioMask;
      return this;
    }

    public Builder airPressure(float pressure) {
      this.airPressure = pressure;
      return this;
    }

    public Builder pirDetectionSeconds(int pirDetectionSeconds){
      this.pirDetectionSeconds = pirDetectionSeconds;
      return this;
    }

    public Builder singleClickCount2(int singleClickCount2){
      this.singleClickCount2 = singleClickCount2;
      return this;
    }

    public Builder manufacturerId(int manufacturerId){
      this.manufacturerId = manufacturerId;
      return this;
    }

    public Builder airQuality(int airQuality){
      this.airQuality = airQuality;
      return this;
    }

    public Builder roomNumber(int roomNumber){
      this.roomNumber = roomNumber;
      return this;
    }

    public Builder occupancy(int occupancy){
      this.occupancy = occupancy;
      return this;
    }

    public Builder occupancyDetectionTime(int occupancyDetectionTime){
      this.occupancyDetectionTime = occupancyDetectionTime;
      return this;
    }

    public KontaktTelemetry build() {
      return new KontaktTelemetry(this);
    }

  }
}
