package com.kontakt.sdk.android.ble.service;

import android.annotation.TargetApi;
import android.app.PendingIntent;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanSettings;
import android.content.Context;
import android.content.Intent;
import android.os.Build;

import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.common.log.Logger;

import java.util.List;
import java.util.concurrent.TimeUnit;

import static com.kontakt.sdk.android.ble.service.ScannerUtil.getBroadcastPendingIntent;
import static com.kontakt.sdk.android.ble.service.ScannerUtil.getReceiverCodeRequest;
import static com.kontakt.sdk.android.ble.service.ScannerUtil.getScanner;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

@SuppressWarnings("MissingPermission")
final class RunnersL {

  private RunnersL() {
    // prevent from creation
  }

  static Runnable newRunner(final Runners.RunnerType runnerType, final ScanConfiguration configuration) {
    switch (runnerType) {
      case MONITOR_ACTIVE_RUNNER:
        return shouldPerformAndroid8Scan((ScanConfigurationL) configuration) ? newMonitorActiveRunnerAndroid8((ScanConfigurationL) configuration) : newMonitorActiveRunnerL((ScanConfigurationL) configuration);
      case MONITOR_PASSIVE_RUNNER:
        return shouldPerformAndroid8Scan((ScanConfigurationL) configuration) ? newMonitorPassiveRunnerAndroid8((ScanConfigurationL) configuration) : newMonitorPassiveRunnerL((ScanConfigurationL) configuration);
      case FORCE_SCAN_RUNNER:
        return newForceScanRunnerL((ScanConfigurationL) configuration);
      default:
        throw new IllegalArgumentException("Invalid runner type passed");
    }
  }

  private static boolean shouldPerformAndroid8Scan(ScanConfigurationL configuration){
    return (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && configuration.getContext() != null);
  }

  private static Runnable newMonitorPassiveRunnerL(final ScanConfigurationL configuration) {
    final MonitorCallbackL monitorCallback = (MonitorCallbackL) configuration.getScanCallback();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.LOLLIPOP)
      @Override
      public void run() {
        final BluetoothLeScanner scanner = getScanner(monitorCallback);
        if (scanner == null) return;

        monitorCallback.onMonitorStopped();
        Logger.d("Stopping scan");
        scanner.stopScan(monitorCallback);
        Logger.d("On monitor stopped");
      }
    };
  }

  private static Runnable newMonitorActiveRunnerL(final ScanConfigurationL configuration) {
    final MonitorCallbackL monitorCallback = (MonitorCallbackL) configuration.getScanCallback();
    final ScanSettings scanSettings = configuration.getScanSettings();
    final List<ScanFilter> scanFilterList = configuration.getScanFilterList();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.LOLLIPOP)
      @Override
      public void run() {
        final BluetoothLeScanner scanner = getScanner(monitorCallback);
        if (scanner == null) return;

        Logger.d("Start Scan");
        scanner.startScan(scanFilterList, scanSettings, monitorCallback);
        Logger.d("On Monitor started");
        monitorCallback.onMonitorStarted();
      }
    };
  }

  private static Runnable newForceScanRunnerL(final ScanConfigurationL configuration) {
    final MonitorCallbackL monitorCallback = (MonitorCallbackL) configuration.getScanCallback();
    final ScanSettings scanSettings = configuration.getScanSettings();

    final ForceScanConfiguration forceScanConfiguration = configuration.getScanContext().getForceScanConfiguration();

    final long forceScanActivePeriod = forceScanConfiguration.getForceScanActivePeriod();
    final long forceScanPassivePeriod = forceScanConfiguration.getForceScanPassivePeriod();

    final List<ScanFilter> scanFilterList = configuration.getScanFilterList();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.LOLLIPOP)
      @Override
      public void run() {
        final BluetoothLeScanner scanner = getScanner(monitorCallback);
        if (scanner == null) return;

        try {
          while (!Thread.currentThread().isInterrupted()) {
            Logger.d("Stopping Scan (force)");
            scanner.stopScan(monitorCallback);

            Logger.d(String.format("Sleep during passive period: %s", String.valueOf(forceScanPassivePeriod)));
            TimeUnit.MILLISECONDS.sleep(forceScanPassivePeriod);

            Logger.d("Starting scan (force)");
            scanner.startScan(scanFilterList, scanSettings, monitorCallback);

            Logger.d(String.format("Sleep during active period: %s", String.valueOf(forceScanActivePeriod)));
            TimeUnit.MILLISECONDS.sleep(forceScanActivePeriod);
          }

          scanner.stopScan(monitorCallback);
          Logger.d("Force scan finished");
        } catch (InterruptedException e) {
          scanner.stopScan(monitorCallback);
          Logger.d("Force scan interrupted");
        }
      }
    };
  }

  @TargetApi(Build.VERSION_CODES.O)
  private static Runnable newMonitorActiveRunnerAndroid8(final ScanConfigurationL configuration){
    checkNotNull(configuration.getContext(), "On Androids 8 and higher, app context cannot be null during scannings");

    final Context context = configuration.getContext();
    final MonitorCallbackL monitorCallback = (MonitorCallbackL) configuration.getScanCallback();
    return new Runnable() {
      @Override
      public void run() {

        monitorCallback.onMonitorStarted();
        final ScanSettings settings = configuration.getScanSettings();
        final List<ScanFilter> filters = configuration.getScanFilterList();

        final BluetoothLeScanner scanner = getScanner(monitorCallback);
        if(scanner == null){
          return;
        }

        int requestCode = getReceiverCodeRequest(monitorCallback);
        BackgroundScanBroadcastReceiver.setMonitorCallbackL(monitorCallback, requestCode);
        Intent intent = new Intent(context, BackgroundScanBroadcastReceiver.class);
        intent.setAction("com.kontakt.sdk.android.ble.service.ACTION_FOUND");

        final PendingIntent pendingIntent = getBroadcastPendingIntent(context, requestCode);
        try {
          int result = scanner.startScan(filters, settings, pendingIntent);
          Logger.d("Result of scanning: " + result);
        } catch(Exception e) {
          Logger.e( "Android 8 startScan() failed");
          e.printStackTrace();
          monitorCallback.onScanFailed(ScanCallback.SCAN_FAILED_INTERNAL_ERROR);
        }

      }
    };
  }

  @TargetApi(Build.VERSION_CODES.O)
  private static Runnable newMonitorPassiveRunnerAndroid8(final ScanConfigurationL configuration){
    checkNotNull(configuration.getContext(), "On Androids 8 and higher, app context cannot be null");
    final Context context = configuration.getContext();

    final MonitorCallbackL monitorCallback = (MonitorCallbackL) configuration.getScanCallback();
    return new Runnable() {
      @Override
      public void run() {
        Logger.d("Unregistering scan receiver");
        monitorCallback.onMonitorStopped();
        int requestCode = getReceiverCodeRequest(monitorCallback);
        BackgroundScanBroadcastReceiver.setMonitorCallbackL(null, requestCode);

        Logger.d("Stopping android 8 background scan");
        BluetoothLeScanner scanner = getScanner(monitorCallback);
        if(scanner == null){
          Logger.d("Background android scan stop: scanner is null, returning");
          return;
        }
        final PendingIntent pendingIntent = getBroadcastPendingIntent(context, requestCode);
        scanner.stopScan(pendingIntent);
      }
    };
  }

}
