package com.kontakt.sdk.android.ble.service;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanRecord;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.os.Build;

import com.kontakt.sdk.android.ble.discovery.ScanErrors;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.io.IOException;
import java.util.Collection;
import java.util.List;

import static com.kontakt.sdk.android.ble.service.ScannerUtil.getScanner;

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
abstract class MonitorCallbackL extends ScanCallback implements BleScanCallback {

  private static final String TAG = MonitorCallbackL.class.getSimpleName();

  private boolean scanStopped = false;

  static MonitorCallbackL wrap(BleScanCallback scanCallback) {
    return new MonitorCallbackL((MonitorCallback) scanCallback) {
    };
  }

  private final MonitorCallback wrappedScanCallback;

  private MonitorCallbackL(MonitorCallback scanCallback) {
    SDKPreconditions.checkNotNull(scanCallback, "Wrapped scan callback is null");
    this.wrappedScanCallback = scanCallback;
  }

  @Override
  public void onScanResult(int callbackType, ScanResult result) {
    if (callbackType != ScanSettings.CALLBACK_TYPE_ALL_MATCHES) {
      return;
    }

    final ScanRecord scanRecord = result.getScanRecord();
    if (scanRecord == null) {
      return;
    }
    onLeScan(result.getDevice(), result.getRssi(), scanRecord.getBytes());
  }

  @Override
  public void onLeScan(BluetoothDevice device, int rssi, byte[] scanRecord) {
    doLeScanIfNotStopped(device, rssi, scanRecord);
  }

  private void doLeScanIfNotStopped(BluetoothDevice device, int rssi, byte[] scanRecord){

    if(scanStopped){
      Logger.w(TAG + " Scan stop was called but scan result came, stopping BT scanner");
      stopScan();
      return;
    }
    Logger.d(TAG +" Scan not stopped yet, processing received bytes");
    wrappedScanCallback.onLeScan(device, rssi, scanRecord);
  }


  @SuppressLint("MissingPermission")
  private void stopScan(){
    try {
      BluetoothLeScanner scanner = getScanner(this);
      if (scanner != null) {
        scanner.stopScan(this);
      }
    } catch (Exception e){
      Logger.e(TAG + " Tried to stop scan from monitor callback, but sth went wrong");
      e.printStackTrace();
    }
  }

  @Override
  public void onScanFailed(int errorCode) {
    final String message = ScanErrors.getMessage(errorCode);
    Logger.e(message);
    wrappedScanCallback.onScanError(new ScanError(message));
  }

  @Override
  public void onBatchScanResults(List<ScanResult> results) {
    Logger.d(TAG + " Batch results arrived: " + results);
    if(results == null){
      Logger.d(TAG + " Batch results null, returning");
      return;
    }
    Logger.d(TAG + " Batch results size: " + results.size());
    for(ScanResult result: results){
      ScanRecord scanRecord = result.getScanRecord();
      if(scanRecord != null) {
        doLeScanIfNotStopped(result.getDevice(), result.getRssi(), scanRecord.getBytes());
      }
      else{
        Logger.w(TAG + " Null scan record");
      }
    }
  }

  @Override
  public Collection<InternalProximityListener> getMonitoringListeners() {
    return wrappedScanCallback.getMonitoringListeners();
  }

  @Override
  public void addListener(InternalProximityListener proximityListener) {
    wrappedScanCallback.addListener(proximityListener);
  }

  @Override
  public void removeListener(InternalProximityListener proximityListener) {
    wrappedScanCallback.removeListener(proximityListener);
  }

  @Override
  public void close() throws IOException {
    wrappedScanCallback.close();
  }

  void onMonitorStarted() {
    scanStopped = false;
    wrappedScanCallback.onMonitorCycleStart();
  }

  void onMonitorStopped() {
    scanStopped = true;
    wrappedScanCallback.onMonitorCycleStop();
  }

}
