package com.kontakt.sdk.android.ble.monitoring;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.kontakt.sdk.android.cloud.adapter.BasicTelemetryCollectEventTypeAdapter;
import com.kontakt.sdk.android.cloud.adapter.FullTelemetryCollectEventTypeAdapter;
import com.kontakt.sdk.android.cloud.api.service.TelemetryCollectEventService;
import com.kontakt.sdk.android.cloud.util.ErrorUtils;
import com.kontakt.sdk.android.common.model.BasicTelemetryCollectEvent;
import com.kontakt.sdk.android.common.model.FullTelemetryCollectEvent;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.io.IOException;
import java.util.List;

import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import retrofit2.converter.scalars.ScalarsConverterFactory;

import static com.kontakt.sdk.android.common.util.ApiUtils.commonHeadersInterceptor;
import static com.kontakt.sdk.android.common.util.ApiUtils.loggingInterceptor;

class EventCollectorClient {

    private final TelemetryCollectEventService eventsService;

    EventCollectorClient(String apiKey, String apiUrl, int apiVersion) {
        SDKPreconditions.checkNotNullOrEmpty(apiKey, "EventCollectorClient - empty api key.");
        SDKPreconditions.checkNotNullOrEmpty(apiUrl, "EventCollectorClient - empty api url.");
        final Retrofit retrofit = buildRetrofit(apiUrl, buildHttpClient(apiKey, apiVersion));
        eventsService = retrofit.create(TelemetryCollectEventService.class);
        ErrorUtils.initialize(retrofit);
    }

    private OkHttpClient buildHttpClient(final String apiKey, final int apiVersion) {
        final OkHttpClient.Builder builder = new OkHttpClient.Builder();
        builder.addInterceptor(commonHeadersInterceptor(apiKey, apiVersion));
        builder.addInterceptor(loggingInterceptor());
        builder.addInterceptor(applicationJsonHeaderInterceptor());
        return builder.build();
    }

    private Interceptor applicationJsonHeaderInterceptor(){
        return new Interceptor() {
            @Override
            public Response intercept(Chain chain) throws IOException {
                final Request original = chain.request();
                final Request.Builder requestBuilder = original.newBuilder()
                        .header("Content-Type", "application/json")
                        .method(original.method(), original.body());

                final Request request = requestBuilder.build();
                return chain.proceed(request);
            }
        };
    }

    private Retrofit buildRetrofit(final String apiBaseUrl, final OkHttpClient okHttpClient) {
        Retrofit.Builder retrofitBuilder = new Retrofit.Builder()
                .baseUrl(apiBaseUrl)
                .addConverterFactory(ScalarsConverterFactory.create())
                .addConverterFactory(GsonConverterFactory.create(gson()))
                .client(okHttpClient);

        return retrofitBuilder.build();
    }

    private Gson gson(){
        return new GsonBuilder()
                .registerTypeAdapter(BasicTelemetryCollectEvent.class, new BasicTelemetryCollectEventTypeAdapter())
                .registerTypeAdapter(FullTelemetryCollectEvent.class, new FullTelemetryCollectEventTypeAdapter())
                .setPrettyPrinting()
                .create();
    }


    void collectBaseEvents(List<BasicTelemetryCollectEvent> collectEventList) throws IOException {
        eventsService.collectBasicTelemetry(collectEventList).execute();
    }

    void collectFullTelemetryEvents(List<FullTelemetryCollectEvent> events) throws IOException  {
        eventsService.collectFullTelemetry(events).execute();
    }
}
