package com.kontakt.sdk.android.ble.monitoring;

import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.spec.KontaktTelemetry;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.BasicTelemetryCollectEvent;
import com.kontakt.sdk.android.common.model.FullTelemetryCollectEvent;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SecureProfileUtils;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public class EventCollector implements IEventCollector {

  private static final String TAG = "EventCollector";

  private final EventSender eventSender;
  private ScanContext scanContext;
  private ScheduledExecutorService executorService;

  public EventCollector(EventSender eventSender, ScanContext scanContext) {
    this.eventSender = checkNotNull(eventSender);
    this.scanContext = checkNotNull(scanContext);
  }

  @Override
  public void collect(RemoteBluetoothDevice device) {
    this.eventSender.collect(device);
  }

  @Override
  public void collectSecureProfile(ISecureProfile secureProfile) {
    this.eventSender.collectSecureProfile(secureProfile);
  }

  @Override
  public void stop() {
    if (executorService == null) {
      return;
    }
    executorService.shutdown();
    executorService = null;
  }

  @Override
  public void clear() {
    this.eventSender.clear();
  }

  @Override
  public void start() {
    if (executorService != null || !scanContext.isMonitoringEnabled()) {
      return;
    }
    executorService = Executors.newSingleThreadScheduledExecutor();
    executorService.scheduleWithFixedDelay(eventSender, scanContext.getMonitoringSyncInterval(), scanContext.getMonitoringSyncInterval(),
        TimeUnit.SECONDS);
  }

  ScheduledExecutorService getExecutorService() {
    return executorService;
  }

  public static class EventSender implements Runnable {

    private static final int DEFAULT_BUFFER_SIZE = 200;
    private final ConcurrentSkipListSet<String> ignored = new ConcurrentSkipListSet<>();
    private final BlockingQueue<BasicTelemetryCollectEvent> basicEventsBuffer;
    private final BlockingQueue<FullTelemetryCollectEvent> fullTelemetryEventsBuffer;
    private final EventCollectorClient eventCollectorClient;
    private final String androidId;

    public EventSender(String androidId) {
      this.basicEventsBuffer = new ArrayBlockingQueue<>(DEFAULT_BUFFER_SIZE, true);
      this.fullTelemetryEventsBuffer = new ArrayBlockingQueue<>(DEFAULT_BUFFER_SIZE, true);
      this.eventCollectorClient = EventCollectorClientFactory.create();
      this.androidId = androidId;
    }

    @Override
    public void run() {
      sendBasicEvents();
      sendFullTelemetryEvents();
    }

    private void sendBasicEvents(){
      final List<BasicTelemetryCollectEvent> events = new ArrayList<>();
      basicEventsBuffer.drainTo(events);
      if (events.isEmpty()) {
        Logger.d(TAG + " Nothing to send from basic events");
        return;
      }

      try {
        eventCollectorClient.collectBaseEvents(events);
      } catch (IOException e) {
        Throwable cause = e.getCause();
        if (cause instanceof UnknownHostException || cause instanceof SocketTimeoutException) {
          basicEventsBuffer.addAll(events);
        } else {
          Logger.e(TAG + " Error occurred when try to send basic monitoring events", e);
        }
      }
    }

    private void sendFullTelemetryEvents(){
      final List<FullTelemetryCollectEvent> events = new ArrayList<>();
      fullTelemetryEventsBuffer.drainTo(events);
      if (events.isEmpty()) {
        Logger.d(TAG + " Nothing to send from full telemetry events");
        return;
      }

      try {
        eventCollectorClient.collectFullTelemetryEvents(events);
      } catch (IOException e) {
        Throwable cause = e.getCause();
        if (cause instanceof UnknownHostException || cause instanceof SocketTimeoutException) {
          fullTelemetryEventsBuffer.addAll(events);
        } else {
          Logger.e(TAG + " Error occurred when try to send full monitoring events", e);
        }
      }
    }

    void collectSecureProfile(ISecureProfile secureProfile) {
      KontaktTelemetry telemetry = secureProfile.getTelemetry();
      if(telemetry != null){
        collectTelemetry(secureProfile);
      } else {
        collect(SecureProfileUtils.asRemoteBluetoothDevice(secureProfile));
      }
    }

    private void collectTelemetry(ISecureProfile secureProfile){
      FullTelemetryCollectEvent event = FullTelemetryCollectEvent.of(secureProfile, androidId);
      final String eventId = event.eventId();
      if (ignored.contains(eventId)) {
        return;
      }
      try {
        fullTelemetryEventsBuffer.add(event);
        ignored.add(eventId);
      } catch (IllegalStateException e) {
        Logger.e("Event collector queue is full", e);
      }
    }

    void collect(final RemoteBluetoothDevice device) {
      final String notificationId = device.getUniqueId() + ":" + device.getBatteryPower();
      if (ignored.contains(notificationId)) {
        return;
      }
      BasicTelemetryCollectEvent event = BasicTelemetryCollectEvent.of(device, androidId);
      try {
        basicEventsBuffer.add(event);
        ignored.add(notificationId);
      } catch (IllegalStateException e) {
        Logger.e("Event collector queue is full", e);
      }
    }

    public void clear() {
      basicEventsBuffer.clear();
      fullTelemetryEventsBuffer.clear();
      ignored.clear();
    }

  }

}
