package com.kontakt.sdk.android.ble.manager.internal;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.Notification;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.provider.Settings;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.ble.manager.service.AbstractServiceConnector;
import com.kontakt.sdk.android.ble.monitoring.EventCollector;
import com.kontakt.sdk.android.ble.monitoring.IEventCollector;
import com.kontakt.sdk.android.ble.service.ProximityService;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.interfaces.SDKBiProvider;
import com.kontakt.sdk.android.common.interfaces.SDKSupplier;
import com.kontakt.sdk.android.common.log.Logger;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

@SuppressWarnings("WeakerAccess")
public class InternalProximityManager extends AbstractServiceConnector {

  static final String TAG = InternalProximityManager.class.getSimpleName();

  private final ShuffledSpacesManager shuffledSpacesManager;
  private final KontaktCloud kontaktCloud;

  private Notification foregroundNotification;
  private int notificationId = -1;

  private InternalProximityListener proximityListener;
  private Context context;
  private ServiceConnection serviceConnection;
  private FutureShufflesCache shufflesCache;
  private IEventCollector eventCollector;
  private Messenger managerMessenger;
  Messenger serviceMessenger;
  boolean isScanning;

  public InternalProximityManager(Context context, KontaktCloud kontaktCloud){
    super(context);
    checkNotNull(kontaktCloud);
    this.context = context.getApplicationContext();
    this.managerMessenger = new Messenger(new ManagerHandler(this));
    this.shuffledSpacesManager = new ShuffledSpacesManager(kontaktCloud);
    this.kontaktCloud = kontaktCloud;
  }

  public void setForegroundNotification(Notification notification, int notificationId){
    this.notificationId = notificationId;
    this.foregroundNotification = notification;
  }

  @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
  @Override
  public synchronized void connect(final OnServiceReadyListener onServiceReadyListener) {
    checkNotNull(onServiceReadyListener, "OnServiceBoundListener is null.");
    checkPermissions();

    if (isConnected()) {
      onServiceReadyListener.onServiceReady();
      Logger.d("ProximityManager already connected to BeaconService.");
      return;
    }

    serviceConnection = new ServiceConnection() {

      @SuppressWarnings("unchecked")
      @Override
      public void onServiceConnected(ComponentName name, IBinder serviceBinder) {
        final SDKSupplier<Messenger> messengerSupplier = (SDKSupplier<Messenger>) serviceBinder;
        final SDKBiProvider<Notification, Integer> notificationProvider = (SDKBiProvider<Notification, Integer>) serviceBinder;
        serviceMessenger = messengerSupplier.get();
        notificationProvider.set(foregroundNotification, notificationId);
        onServiceReadyListener.onServiceReady();
        Logger.d("Proximity Service connected.");
      }

      @Override
      public void onServiceDisconnected(ComponentName name) {
        Logger.d("Proximity Service disconnected.");
      }
    };

    bindService();
  }

  @Override
  public synchronized void disconnect() {
    if (!isConnected()) {
      Logger.d(TAG + ": ProximityManager already disconnected.");
      return;
    } else if (isScanning()) {
      finishScan();
    }

    try {
      if (serviceConnection != null) {
        final Message message = createMessage(ProximityService.MESSAGE_WORK_FINISHED, null);
        serviceMessenger.send(message);
        context.unbindService(serviceConnection);
        serviceConnection = null;
        serviceMessenger = null;
      }
      super.disconnect();
    } catch (RemoteException e) {
      Logger.e(TAG + ": unexpected exception thrown while disconnecting from ProximityService ", e);
      throw new IllegalStateException(e);
    }

    shuffledSpacesManager.onDestroy();
  }

  @Override
  public synchronized boolean isConnected() {
    return serviceConnection != null && serviceMessenger != null;
  }

  public synchronized void initializeScan(final ScanContext scanContext, final InternalProximityListener proximityListener) {
    checkNotNull(scanContext, "ScanContext cannot be null");
    checkNotNull(proximityListener, "InternalProximityListener cannot be null");
    Logger.d("Initializing scan...");
    resolveShuffledSpaces(scanContext, proximityListener, false);
  }

  public synchronized void restartScan(final ScanContext scanContext, final InternalProximityListener proximityListener) {
    checkNotNull(scanContext, "ScanContext cannot be null");
    checkNotNull(proximityListener, "InternalProximityListener cannot be null");
    resolveShuffledSpaces(scanContext, proximityListener, true);
  }

  public synchronized void finishScan() {
    Logger.d("Finishing scan...");

    if (!isConnected()) {
      Logger.d("ProximityManager not connected");
      return;
    }

    if (!isScanning()) {
      Logger.d("ProximityManager is not scanning");
      return;
    }

    detachListener();
    final Message message = createMessage(ProximityService.MESSAGE_FINISH_SCAN, null);
    sendMessage(message);

    if (proximityListener != null) {
      proximityListener.onScanStop();
    }
  }

  public synchronized boolean isScanning() {
    return isScanning;
  }

  private synchronized void attachListener(final InternalProximityListener proximityListener) {
    checkNotNull(proximityListener, "Proximity listener is null");
    this.proximityListener = proximityListener;
    final Message message = createMessage(ProximityService.MESSAGE_ATTACH_MONITORING_LISTENER, proximityListener);
    sendMessage(message);
  }

  private synchronized void detachListener() {
    if (proximityListener != null) {
      final Message message = createMessage(ProximityService.MESSAGE_DETACH_MONITORING_LISTENER, proximityListener);
      sendMessage(message);
    }
  }

  private void bindService() {
    Logger.d("Binding to ProximityService...");
    final Intent serviceIntent = new Intent(context, ProximityService.class);
    final boolean isBindRequestSent = context.bindService(serviceIntent, serviceConnection, Context.BIND_AUTO_CREATE);
    if (!isBindRequestSent) {
      final String serviceName = ProximityService.class.getSimpleName();
      throw new RuntimeException(
          String.format("Could not connect to %s. Please check if the %s is registered in AndroidManifest.xml", serviceName, serviceName));
    }
  }

  private void resolveShuffledSpaces(ScanContext scanContext, final InternalProximityListener proximityListener, final boolean restartScan) {
    Logger.d("Resolving shuffled spaces...");
    shuffledSpacesManager.resolve(scanContext, new ShuffledSpacesManager.OnSpacesResolvedListener() {
      @Override
      public void onSpacesResolved(ScanContext scanContext) {
        Logger.d("Shuffled spaces resolved.");
        startScanIfConnected(scanContext, restartScan, proximityListener);
      }

      @Override
      public void onError(ScanError exception) {
        Logger.e("Shuffled spaces resolving error: " + exception.getMessage());
        proximityListener.onScanError(exception);
      }
    });
  }

  void startScanIfConnected(final ScanContext scanContext, boolean restartScan, InternalProximityListener proximityListener) {
    if (isScanning() && !restartScan) {
      Logger.d("ProximityManager is already scanning.");
      return;
    }

    shufflesCache = new FutureShufflesCache(context, kontaktCloud, scanContext);
    startEventCollector(scanContext);
    ProximityService.Bundle bundle = new ProximityService.Bundle(scanContext, shufflesCache, eventCollector);

    int messageCode = restartScan ? ProximityService.MESSAGE_RESTART_SCAN : ProximityService.MESSAGE_INITIALIZE_SCAN;
    final Message message = createMessage(messageCode, bundle);
    sendMessage(message);

    attachListener(proximityListener);
    proximityListener.onScanStart();
  }

  private boolean sendMessage(final Message message) {
    if (isConnected()) {
      try {
        serviceMessenger.send(message);
        return true;
      } catch (RemoteException e) {
        return false;
      }
    } else {
      Logger.d("ProximityManager already disconnected");
    }

    return false;
  }

  private Message createMessage(final int messageCode, final Object obj) {
    final Message message = Message.obtain(null, messageCode, -1, -1, obj);
    message.replyTo = managerMessenger;
    return message;
  }

  private void startEventCollector(ScanContext scanContext) {
    if (eventCollector != null) {
      eventCollector.stop();
    }
    eventCollector = new EventCollector(new EventCollector.EventSender(getAndroidId()), scanContext);
    eventCollector.start();
  }

  @SuppressLint("HardwareIds")
  private String getAndroidId(){
    return Settings.Secure.getString(context.getContentResolver(),
            Settings.Secure.ANDROID_ID);
  }

  //Used internally
  public synchronized void clearCache() {
    Logger.d("Clearing cache...");
    if (shufflesCache != null) {
      shufflesCache.clear();
    }
    clearEventCollector();
  }

  //Used internally
  public synchronized void clearBuffers() {
    Logger.d("Clearing buffers...");
    if (shufflesCache != null) {
      shufflesCache.clearBuffers();
    }
    clearEventCollector();
  }

  private void clearEventCollector() {
    Logger.d("Clearing event collector...");
    if (eventCollector != null) {
      eventCollector.clear();
    }
  }

  private static final class ManagerHandler extends Handler {

    private final InternalProximityManager manager;

    ManagerHandler(InternalProximityManager manager) {
      this.manager = manager;
    }

    @Override
    public void handleMessage(Message msg) {
      switch (msg.what) {
        case ProximityService.MESSAGE_INITIALIZE_SCAN:
        case ProximityService.MESSAGE_RESTART_SCAN:
          manager.isScanning = (msg.arg1 == ProximityService.MESSAGE_SERVICE_RESPONSE_OK);
          break;

        case ProximityService.MESSAGE_FINISH_SCAN:
        case ProximityService.MESSAGE_WORK_FINISHED:
          manager.isScanning = false;
          break;

        default:
          throw new IllegalArgumentException("Unsupported response code: " + msg.what);
      }
    }
  }
}
