package com.kontakt.sdk.android.ble.manager.configuration;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.KontaktScanFilter;
import com.kontakt.sdk.android.ble.configuration.ScanMode;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.discovery.secure_profile.PayloadResolver;
import com.kontakt.sdk.android.ble.manager.internal.InternalProximityManager;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;

import java.util.Collection;
import java.util.List;

public interface GeneralConfigurator {

  /**
   * Sets {@link ScanPeriod}. Default: {@link ScanPeriod#RANGING}
   *
   * @param scanPeriod ScanPeriod
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator scanPeriod(ScanPeriod scanPeriod);

  /**
   * Sets {@link ScanMode}. Default: {@link ScanMode#BALANCED}
   *
   * @param scanMode ScanMode
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator scanMode(ScanMode scanMode);

  /**
   * Sets {@link ForceScanConfiguration}. This will always be set to DISABLED for Android N and higher (https://github.com/AltBeacon/android-beacon-library/issues/418).
   * <br>
   * Default: {@link ForceScanConfiguration#DISABLED}
   *
   * @param forceScanConfiguration ForceScanConfiguration
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator forceScanConfiguration(ForceScanConfiguration forceScanConfiguration);

  /**
   * Sets {@link ActivityCheckConfiguration}. Default: {@link ActivityCheckConfiguration#DEFAULT}
   *
   * @param activityCheckConfiguration ActivityCheckConfiguration.
   * @return GeneralConfigurator builder.
   */
  GeneralConfigurator activityCheckConfiguration(ActivityCheckConfiguration activityCheckConfiguration);

  /**
   * Cache file name for resolved shuffled device ids from Kontakt.io Proximity REST API.
   * Cache file name must be unique for safe serialization between different {@link InternalProximityManager} used in the same app.
   *
   * @param name file name
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator cacheFileName(String name);

  /**
   * Enables or disables collecting monitoring events e.g. battery level of device. Default: ENABLED.
   *
   * @param enabled true or false
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator monitoringEnabled(boolean enabled);

  /**
   * Interval between next monitoring events sync attempts in seconds. Must be greater than 0. Default: 10 seconds.
   *
   * @param intervalInSeconds interval in seconds
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator monitoringSyncInterval(int intervalInSeconds);

  /**
   * Interval between next shuffled beacons resolving attempts in seconds. Must be greater than 0. Default: 3 seconds.
   *
   * @param intervalInSeconds interval in seconds
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator resolveShuffledInterval(int intervalInSeconds);

  /**
   * The device update callback interval prevents the update from being called instantly.
   * <br>
   * Once the Android device enters a space with many BLE devices the update event starts to be called with instantly growing number of
   * updated BLE devices.
   * <br>
   * This may cause interruptions when interacting with UI. The device update callback interval option causes the update callback to
   * be invoked no more often than given interval. Default: 3000 milliseconds.
   *
   * @param intervalInMillis Interval in milliseconds.
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator deviceUpdateCallbackInterval(long intervalInMillis);

  /**
   * Sets {@link RssiCalculator}.
   *
   * @param rssiCalculator RSSI calculator instance.
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator rssiCalculator(RssiCalculator rssiCalculator);

  /**
   * Sets trigger frame types.
   * The trigger frame types specify with which frame types received at least the Eddystone device
   * should be notified.
   * <br>
   * If not set Eddystone device will be notified as soon as any frame will be scanned (UID, URL or TLM).
   *
   * @param eddystoneFrameTypes eddystone format frame types
   * @return GeneralConfigurator builder
   */
  GeneralConfigurator eddystoneFrameTypes(Collection<EddystoneFrameType> eddystoneFrameTypes);

  /**
   * Sets custom payload resolver that is used internally to parse payload and resolve secure profile object from it.
   * Resolving algorithm is provided by user's implementation.
   *
   * @param payloadResolver the resolver implementation.
   * @return GeneralConfigurator builder.
   */
  GeneralConfigurator secureProfilePayloadResolver(PayloadResolver payloadResolver);

  /**
   * Sets custom payload resolvers that are used internally to parse payloads and resolve secure profiles object.
   * Resolving algorithms are provided by user's implementation.
   *
   * @param payloadResolvers the resolvers collection.
   * @return GeneralConfigurator builder.
   */
  GeneralConfigurator secureProfilePayloadResolvers(Collection<PayloadResolver> payloadResolvers);

  GeneralConfigurator kontaktScanFilters(List<KontaktScanFilter> filterList);

}
