package com.kontakt.sdk.android.ble.manager;

import android.app.Notification;
import android.content.BroadcastReceiver;
import android.content.Context;

import android.content.Intent;
import android.content.IntentFilter;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;
import com.kontakt.sdk.android.ble.manager.configuration.FiltersConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.GeneralConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.SpacesConfigurator;
import com.kontakt.sdk.android.ble.manager.internal.InternalProximityManager;
import com.kontakt.sdk.android.ble.manager.listeners.EddystoneListener;
import com.kontakt.sdk.android.ble.manager.listeners.IBeaconListener;
import com.kontakt.sdk.android.ble.manager.listeners.ScanStatusListener;
import com.kontakt.sdk.android.ble.manager.listeners.SecureProfileListener;
import com.kontakt.sdk.android.ble.manager.listeners.SpaceListener;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.KontaktCloudFactory;
import com.kontakt.sdk.android.common.log.Logger;

import static com.kontakt.sdk.android.common.profile.DeviceProfile.EDDYSTONE;
import static com.kontakt.sdk.android.common.profile.DeviceProfile.IBEACON;
import static com.kontakt.sdk.android.common.profile.DeviceProfile.KONTAKT_SECURE;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public class ProximityManagerImpl implements ProximityManager {

  static final String BLE_SCAN_ERROR_OCCURRED_ACTION = "com.kontakt.sdk.action.BLE_SCAN_ERROR_OCCURRED";

  private final Context context;

  private IBeaconListener iBeaconListener;
  private EddystoneListener eddystoneListener;
  private SecureProfileListener secureProfileListener;
  private ScanStatusListener scanStatusListener;
  private SpaceListener spaceListener;
  private InternalProximityManager internalProximityManager;
  private ScanContext.Builder scanContext = new ScanContext.Builder();

  ProximityManagerImpl(Context context) {
    this(context, KontaktCloudFactory.create());
  }

  ProximityManagerImpl(Context context, KontaktCloud kontaktCloud) {
    checkNotNull(context, "Context can't be null").getApplicationContext();
    checkNotNull(kontaktCloud, "Kontakt Cloud can't be null");
    this.context = context;
    this.internalProximityManager = new InternalProximityManager(context.getApplicationContext(), kontaktCloud);
  }

  @Override
  public void connect(OnServiceReadyListener onServiceReadyListener) {
    checkNotNull(onServiceReadyListener, "OnServiceReadyListener can't be null.");
    if (!internalProximityManager.isConnected()) {
      internalProximityManager.connect(onServiceReadyListener);
      registerBleScanErrorOccurredReceiver();
    } else {
      onServiceReadyListener.onServiceReady();
      Logger.d("ProximityManager is already connected.");
    }
  }

  @Override
  public void disconnect() {
    internalProximityManager.disconnect();
    unregisterBleScanErrorOccurredReceiver();
  }

  @Override
  public void startScanning() {
    checkArgument(internalProximityManager.isConnected(),
        "ProximityManager is not connected to ProximityService. Use ProximityManager.connect() before starting a scan.");
    if (!internalProximityManager.isScanning()) {
      checkObservedProfiles();
      internalProximityManager.initializeScan(scanContext.build(), createEventObserver());
    }
  }

  @Override
  public void restartScanning() {
    checkArgument(internalProximityManager.isConnected(),
        "ProximityManager is not connected to ProximityService. Use ProximityManager.connect() before starting a scan.");
    if (internalProximityManager.isScanning()) {
      checkObservedProfiles();
      internalProximityManager.restartScan(scanContext.build(), createEventObserver());
    }
  }

  @Override
  public void stopScanning() {
    if (internalProximityManager.isScanning()) {
      internalProximityManager.finishScan();
    }
  }

  @Override
  public boolean isConnected() {
    return internalProximityManager.isConnected();
  }

  @Override
  public boolean isScanning() {
    return internalProximityManager.isScanning();
  }

  @Override
  public void setScanStatusListener(ScanStatusListener listener) {
    this.scanStatusListener = listener;
  }

  @Override
  public void setSpaceListener(SpaceListener listener) {
    this.spaceListener = listener;
  }

  @Override
  public void setIBeaconListener(IBeaconListener listener) {
    this.iBeaconListener = listener;
  }

  @Override
  public void setSecureProfileListener(SecureProfileListener listener) {
    this.secureProfileListener = listener;
  }

  @Override
  public void setEddystoneListener(EddystoneListener listener) {
    this.eddystoneListener = listener;
  }

  @Override
  public GeneralConfigurator configuration() {
    return scanContext;
  }

  @Override
  public SpacesConfigurator spaces() {
    return scanContext;
  }

  @Override
  public FiltersConfigurator filters() {
    return scanContext;
  }

  @Override
  public void setForegroundNotification(Notification notification, int notificationId) {
    this.internalProximityManager.setForegroundNotification(notification, notificationId);
  }

  @Override
  public void clearForegroundNotification() {
    this.internalProximityManager.setForegroundNotification(null, -1);
  }

  private void checkObservedProfiles() {
    if (iBeaconListener == null) {
      scanContext.removeObservedProfile(IBEACON);
    } else {
      scanContext.addObservedProfile(IBEACON);
    }

    if (eddystoneListener == null) {
      scanContext.removeObservedProfile(EDDYSTONE);
    } else {
      scanContext.addObservedProfile(EDDYSTONE);
    }

    if (secureProfileListener == null) {
      scanContext.removeObservedProfile(KONTAKT_SECURE);
    } else {
      scanContext.addObservedProfile(KONTAKT_SECURE);
    }
  }

  private EventObserver createEventObserver() {
    return new EventObserver(context, eddystoneListener, iBeaconListener, scanStatusListener, spaceListener, secureProfileListener);
  }

  private void registerBleScanErrorOccurredReceiver() {
    final IntentFilter intentFilter = new IntentFilter();
    intentFilter.addAction(BLE_SCAN_ERROR_OCCURRED_ACTION);
    context.registerReceiver(bleScanErrorOccurredReceiver, intentFilter);
  }

  private void unregisterBleScanErrorOccurredReceiver() {
    try {
      context.unregisterReceiver(bleScanErrorOccurredReceiver);
    } catch (IllegalArgumentException e) {
      // ignore
    }
  }

  private final BroadcastReceiver bleScanErrorOccurredReceiver = new BroadcastReceiver() {
    @Override
    public void onReceive(Context context, Intent intent) {
      disconnect();
    }
  };

}
