package com.kontakt.sdk.android.ble.discovery.secure_profile;

import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.ble.discovery.FramePayload;
import com.kontakt.sdk.android.common.model.Model;
import java.util.Arrays;
import java.util.Locale;

import static com.kontakt.sdk.android.common.util.ConversionUtils.asInt;

public class PlainDevicePayloadResolver extends PayloadResolver {

  private static final byte PAYLOAD_IDENTIFIER = 0x02;

  public PlainDevicePayloadResolver() {
    super(PAYLOAD_IDENTIFIER);
  }

  @Override
  public void resolve(final FramePayload payload) {
    if (payload == null) {
      return;
    }

    // Get MAC address, name & service data
    final String macAddress = payload.getAddress();
    final String name = getName(payload);
    final byte[] serviceData = getServiceData(payload);

    // Parse properties
    final Model model = Model.fromCode(serviceData[3]);
    final String firmware = parseFirmwareVersion(serviceData);
    final int batteryLevel = asInt(serviceData[6]);
    final int txPower = serviceData[7];
    final String uniqueId = parseUniqueId(serviceData);

    // Build plain secure profile object
    final SecureProfile secureProfile = new SecureProfile.Builder()
        .macAddress(macAddress)
        .name(name)
        .model(model)
        .firmwareRevision(firmware)
        .batteryLevel(batteryLevel)
        .txPower(txPower)
        .uniqueId(uniqueId)
        .shuffled(false)
        .build();

    // Do callback
    onPayloadResolved(secureProfile);
  }

  private String parseFirmwareVersion(final byte[] serviceData) {
    final int firmwareVersionMajor = serviceData[4];
    final int firmwareVersionMinor = serviceData[5];

    if (firmwareVersionMajor < 0 || firmwareVersionMinor < 0) {
      return "";
    }
    return String.format(Locale.getDefault(), "%d.%d", firmwareVersionMajor, firmwareVersionMinor);
  }

  private String parseUniqueId(final byte[] serviceData) {
    return new String(Arrays.copyOfRange(serviceData, 8, serviceData.length));
  }

}
