package com.kontakt.sdk.android.ble.discovery;

import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import java.util.Arrays;

public final class ScanResponse {

  public static final ScanResponse UNKNOWN = new ScanResponse(null, -1, Constants.Devices.FIRMWARE_VERSION_NONE, false, new byte[0]);

  public static final int TYPE_SERVICE_DATA = 0x16;
  public static final byte PAYLOAD_VERSION_1 = 1;
  public static final byte D00D_SERVICE_UUID_MSB = 0x0D;
  public static final byte D00D_SERVICE_UUID_LSB = (byte) 0xD0;
  public static final byte PDI_SERVICE_UUID_MSB = 0x6A;
  public static final byte PDI_SERVICE_UUID_LSB = (byte) 0xFE;
  public static final int KONTAKT_DEVICE_SCAN_RESPONSE_MIN_LENGTH = 9;

  public static final byte[] SCAN_RESPONSE_D00D_PREFIX = new byte[] { D00D_SERVICE_UUID_MSB, D00D_SERVICE_UUID_LSB };
  public static final byte[] SCAN_RESPONSE_PDI_PREFIX = new byte[] { PDI_SERVICE_UUID_MSB, PDI_SERVICE_UUID_LSB };

  private final String uniqueId;
  private final int batteryPercentagePower;
  private final String firmwareVersion;
  private final boolean shuffled;
  private final byte[] bytes;

  public static boolean isValidKontaktScanResponse(byte[] serviceData) {
    return serviceData.length >= KONTAKT_DEVICE_SCAN_RESPONSE_MIN_LENGTH && ConversionUtils.doesArrayBeginWith(serviceData,
        SCAN_RESPONSE_D00D_PREFIX);
  }

    public static ScanResponse fromScanResponseBytes(final byte[] scanResponse) {
    if (scanResponse == null || !isValidKontaktScanResponse(scanResponse)) {
      return UNKNOWN;
    }

    String uniqueId = null;
    String firmware;
    int batteryPercentagePower;

    byte payloadVersion = scanResponse[2];
    boolean isShuffled = payloadVersion == PAYLOAD_VERSION_1;
    if (isShuffled) {
      firmware = parseShuffledFirmwareVersion(scanResponse);
      batteryPercentagePower = extractShuffledBatteryPercentagePower(scanResponse);
    } else {
      uniqueId = parseUniqueId(scanResponse);
      firmware = parseFirmwareVersion(scanResponse);
      batteryPercentagePower = extractBatteryPercentagePower(scanResponse);
    }

    return new ScanResponse(uniqueId, batteryPercentagePower, firmware, isShuffled, scanResponse);
  }

  private ScanResponse(String uniqueId, int batteryPercentagePower, String firmwareVersion, boolean shuffled, byte[] bytes) {
    this.uniqueId = uniqueId;
    this.batteryPercentagePower = batteryPercentagePower;
    this.firmwareVersion = firmwareVersion;
    this.shuffled = shuffled;
    this.bytes = bytes;
  }

  public String getFirmwareVersion() {
    return firmwareVersion;
  }

  public int getBatteryPower() {
    return batteryPercentagePower;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  public byte[] getBytes() {
    return bytes;
  }

  public boolean isShuffled() {
    return shuffled;
  }

  public boolean isUnknown() {
    return uniqueId == null && batteryPercentagePower == -1 && Constants.Devices.FIRMWARE_VERSION_NONE.equalsIgnoreCase(firmwareVersion);
  }

  private static String parseShuffledFirmwareVersion(final byte[] serviceData) {
    if (serviceData != null) {
      char firmwareVersionMajor = (char) (serviceData[23] & 0x00FF);
      char firmwareVersionMinor = (char) (serviceData[24] & 0x00FF);
      return String.format("%c.%c", firmwareVersionMajor, firmwareVersionMinor);
    }

    return Constants.Devices.FIRMWARE_VERSION_NONE;
  }

  private static String parseFirmwareVersion(final byte[] serviceData) {
    if (serviceData != null) {
      final char firmwareVersionMajor = (char) (serviceData[6] & 0x00FF);
      final char firmwareVersionMinor = (char) (serviceData[7] & 0x00FF);
      return String.format("%c.%c", firmwareVersionMajor, firmwareVersionMinor);
    }

    return Constants.Devices.FIRMWARE_VERSION_NONE;
  }

  private static int extractBatteryPercentagePower(final byte[] serviceData) {
    return serviceData != null ? serviceData[8] : -1;
  }

  private static int extractShuffledBatteryPercentagePower(final byte[] serviceData) {
    return serviceData != null ? serviceData[25] : -1;
  }

  private static String parseUniqueId(final byte[] serviceData) {
    return serviceData != null ? new String(Arrays.copyOfRange(serviceData, 2, 6)) : "";
  }

    @Override
    public String toString() {
        return "Scan response: \n" +
                "uniqueId: " + uniqueId + "\n" +
                "battery percentage: " + batteryPercentagePower + "\n" +
                "firmware version: " +  firmwareVersion + "\n" +
                "shuffled: " + shuffled;
    }
}
