package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import android.text.TextUtils;

import com.kontakt.sdk.android.ble.spec.Telemetry;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import com.kontakt.sdk.android.common.model.Model;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

public class EddystoneDevice implements IEddystoneDevice {

  public static final Creator<EddystoneDevice> CREATOR = new Creator<EddystoneDevice>() {
    @Override
    public EddystoneDevice createFromParcel(Parcel source) {
      return new EddystoneDevice(source);
    }

    @Override
    public EddystoneDevice[] newArray(int size) {
      return new EddystoneDevice[size];
    }
  };

  private final HashCodeBuilder hashCodeBuilder;

  private String address;
  private String namespace;
  private String instanceId;
  private String url;
  private Telemetry telemetry;
  private String encryptedTelemetry;
  private String eid;
  private String uniqueId;
  private String name;
  private String firmwareVersion;
  private int txPower;
  private int batteryPower;
  private boolean shuffled;
  private Proximity proximity;
  private long timestamp;
  private double distance;
  private int rssi;
  private byte[] password;

  EddystoneDevice(Parcel source) {
    Bundle bundle = source.readBundle(getClass().getClassLoader());

    this.name = bundle.getString(Constants.Devices.NAME);
    this.timestamp = bundle.getLong(Constants.TIMESTAMP);
    this.namespace = bundle.getString(Constants.Eddystone.NAMESPACE_ID);
    this.instanceId = bundle.getString(Constants.Eddystone.INSTANCE_ID);
    this.txPower = bundle.getInt(Constants.Devices.TX_POWER);
    this.url = bundle.getString(Constants.Eddystone.URL);
    this.distance = bundle.getDouble(Constants.Devices.ACCURACY);
    this.address = bundle.getString(Constants.Devices.ADDRESS);
    this.proximity = (Proximity) bundle.getSerializable(Constants.Devices.PROXIMITY);
    this.rssi = bundle.getInt(Constants.Devices.RSSI);
    this.telemetry = bundle.getParcelable(Constants.Eddystone.TELEMETRY);
    this.encryptedTelemetry = bundle.getString(Constants.Eddystone.ENCRYPTED_TELEMETRY);
    this.eid = bundle.getString(Constants.Eddystone.EID);
    this.password = bundle.getByteArray(Constants.Devices.PASSWORD);
    this.firmwareVersion = bundle.getString(Constants.FIRMWARE);
    this.uniqueId = bundle.getString(Constants.UNIQUE_ID);
    this.batteryPower = bundle.getInt(Constants.Devices.BATTERY);
    this.shuffled = bundle.getBoolean(Constants.Devices.SHUFFLED);
    this.hashCodeBuilder = HashCodeBuilder.init();
  }

  EddystoneDevice(final Builder builder) {
    this.name = builder.name;
    this.timestamp = builder.timestamp;
    this.namespace = builder.namespace;
    this.instanceId = builder.instanceId;
    this.txPower = builder.txPower;
    this.url = builder.url;
    this.distance = builder.distance;
    this.address = builder.address;
    this.proximity = builder.proximity;
    this.rssi = builder.rssi;
    this.telemetry = builder.telemetry;
    this.encryptedTelemetry = builder.encryptedTelemetry;
    this.eid = builder.eid;
    this.firmwareVersion = builder.firmwareVersion;
    this.uniqueId = builder.uniqueId;
    this.batteryPower = builder.batteryPower;
    this.shuffled = builder.shuffled;
    this.password = builder.password;
    this.hashCodeBuilder = HashCodeBuilder.init();
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public int hashCode() {
    return hashCodeBuilder.append(address).build();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || !(o instanceof EddystoneDevice)) {
      return false;
    }

    final EddystoneDevice device = (EddystoneDevice) o;

    return TextUtils.equals(address, device.address);
  }

  @Override
  public String toString() {
    return "EddystoneDevice{" +
        "address='" + address + '\'' +
        ", uniqueId='" + uniqueId + '\'' +
        ", namespace='" + namespace + '\'' +
        ", instanceId='" + instanceId + '\'' +
        ", url='" + url + '\'' +
        ", eid='" + eid + '\'' +
        ", etlm='" + encryptedTelemetry + '\'' +
        ", rssi=" + rssi +
        ", shuffled=" + shuffled +
        '}';
  }

  @Override
  public int compareTo(IEddystoneDevice another) {
    if (this == another) {
      return 0;
    }

    final int namespaceComparisonResult = namespace.compareTo(another.getNamespace());

    if (namespaceComparisonResult == 0) {

      final int instanceComparisonResult = instanceId.compareTo(another.getInstanceId());

      if (instanceComparisonResult == 0) {

        return url.compareTo(another.getUrl());
      } else {
        return instanceComparisonResult;
      }
    } else {
      return namespaceComparisonResult;
    }
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    final Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putString(Constants.Eddystone.NAMESPACE_ID, namespace);
    bundle.putString(Constants.Eddystone.INSTANCE_ID, instanceId);
    bundle.putString(Constants.Eddystone.URL, url);
    bundle.putInt(Constants.Devices.TX_POWER, txPower);
    bundle.putLong(Constants.TIMESTAMP, timestamp);
    bundle.putDouble(Constants.Devices.ACCURACY, distance);
    bundle.putSerializable(Constants.Devices.PROXIMITY, proximity);
    bundle.putInt(Constants.Devices.RSSI, rssi);
    bundle.putString(Constants.Devices.ADDRESS, address);
    bundle.putParcelable(Constants.Eddystone.TELEMETRY, telemetry);
    bundle.putString(Constants.Eddystone.ENCRYPTED_TELEMETRY, encryptedTelemetry);
    bundle.putString(Constants.Eddystone.EID, eid);
    bundle.putByteArray(Constants.Devices.PASSWORD, password);
    bundle.putString(Constants.Devices.NAME, name);
    bundle.putString(Constants.FIRMWARE, firmwareVersion);
    bundle.putString(Constants.UNIQUE_ID, uniqueId);
    bundle.putInt(Constants.Devices.BATTERY, batteryPower);
    bundle.putBoolean(Constants.Devices.SHUFFLED, shuffled);
    dest.writeBundle(bundle);
  }

  @Override
  public double getDistance() {
    return distance;
  }

  @Override
  public long getTimestamp() {
    return timestamp;
  }

  @Override
  public String getAddress() {
    return address;
  }

  @Override
  public Proximity getProximity() {
    return proximity;
  }

  @Override
  public int getRssi() {
    return rssi;
  }

  @Override
  public byte[] getPassword() {
    return password;
  }

  @Override
  public void setPassword(byte[] password) {
    SDKPreconditions.checkNotNull(password, "Password is null");
    final int passwordLength = password.length;
    this.password = new byte[passwordLength];
    System.arraycopy(password, 0, this.password, 0, passwordLength);
  }

  @Override
  public Model getModel() {
    return Model.UNKNOWN;
  }

  @Override
  public String getFirmwareVersion() {
    return firmwareVersion;
  }

  @Override
  public String getName() {
    return name;
  }

  @Override
  public String getUniqueId() {
    return uniqueId;
  }

  @Override
  public int getBatteryPower() {
    return batteryPower;
  }

  @Override
  public String getNamespace() {
    return namespace;
  }

  public void setNamespace(String namespace) {
    this.namespace = namespace;
  }

  @Override
  public String getInstanceId() {
    return instanceId;
  }

  public void setInstanceId(String instanceId) {
    this.instanceId = instanceId;
  }

  @Override
  public int getTxPower() {
    return txPower;
  }

  @Override
  public DeviceProfile getProfile() {
    return DeviceProfile.EDDYSTONE;
  }

  @Override
  public String getUrl() {
    return url;
  }

  public void setUrl(String url) {
    this.url = url;
  }

  @Override
  public Telemetry getTelemetry() {
    return telemetry;
  }

  public void setTelemetry(Telemetry telemetry) {
    this.telemetry = telemetry;
  }

  @Override
  public String getEncryptedTelemetry() {
    return encryptedTelemetry;
  }

  public void setEncryptedTelemetry(String encryptedTelemetry) {
    this.encryptedTelemetry = encryptedTelemetry;
  }

  @Override
  public String getEid() {
    return eid;
  }

  public void setEid(String eid) {
    this.eid = eid;
  }

  @Override
  public boolean isShuffled() {
    return shuffled;
  }

  public static class Builder {

    String namespace;
    String instanceId;
    int txPower;
    String url;
    double distance;
    String address;
    Proximity proximity;
    int rssi;
    long timestamp;
    String firmwareVersion;
    String uniqueId;
    int batteryPower;
    String name;
    boolean shuffled;
    Telemetry telemetry;
    String encryptedTelemetry;
    String eid;
    byte[] password;

    public Builder() {
    }

    public Builder(IEddystoneDevice eddystoneDevice) {
      this.name = eddystoneDevice.getName();
      this.timestamp = eddystoneDevice.getTimestamp();
      this.namespace = eddystoneDevice.getNamespace();
      this.instanceId = eddystoneDevice.getInstanceId();
      this.txPower = eddystoneDevice.getTxPower();
      this.url = eddystoneDevice.getUrl();
      this.distance = eddystoneDevice.getDistance();
      this.address = eddystoneDevice.getAddress();
      this.proximity = eddystoneDevice.getProximity();
      this.rssi = eddystoneDevice.getRssi();
      if (eddystoneDevice.getTelemetry() != null) {
        this.telemetry = new Telemetry.Builder().batteryVoltage(eddystoneDevice.getTelemetry().getBatteryVoltage())
            .temperature(eddystoneDevice.getTelemetry().getTemperature())
            .pduCount(eddystoneDevice.getTelemetry().getPduCount())
            .timeSincePowerUp(eddystoneDevice.getTelemetry().getTimeSincePowerUp())
            .version(eddystoneDevice.getTelemetry().getVersion())
            .build();
      }
      this.encryptedTelemetry = eddystoneDevice.getEncryptedTelemetry();
      this.eid = eddystoneDevice.getEid();
      this.firmwareVersion = eddystoneDevice.getFirmwareVersion();
      this.uniqueId = eddystoneDevice.getUniqueId();
      this.batteryPower = eddystoneDevice.getBatteryPower();
      this.shuffled = eddystoneDevice.isShuffled();
    }

    public Builder resolvedId(ResolvedId resolvedId) {
      EddystoneUid eddystoneUID = resolvedId.getEddystoneUID();
      this.namespace = eddystoneUID.getNamespace();
      this.instanceId = eddystoneUID.getInstanceId();
      this.uniqueId = resolvedId.getUniqueId();
      return this;
    }

    public Builder namespace(String namespace) {
      this.namespace = namespace;
      return this;
    }

    public Builder instanceId(String instanceId) {
      this.instanceId = instanceId;
      return this;
    }

    public Builder url(String url) {
      this.url = url;
      return this;
    }

    public Builder telemetry(Telemetry telemetry) {
      this.telemetry = telemetry;
      return this;
    }

    public Builder encryptedTelemetry(String encryptedTelemetry) {
      this.encryptedTelemetry = encryptedTelemetry;
      return this;
    }

    public Builder eid(String eid) {
      this.eid = eid;
      return this;
    }

    public Builder txPower(int txPower) {
      this.txPower = txPower;
      return this;
    }

    public Builder distance(double distance) {
      this.distance = distance;
      return this;
    }

    public Builder address(String address) {
      this.address = address;
      return this;
    }

    public Builder proximity(Proximity proximity) {
      this.proximity = proximity;
      return this;
    }

    public Builder rssi(int rssi) {
      this.rssi = rssi;
      return this;
    }

    public Builder firmwareRevision(String firmwareVersion) {
      this.firmwareVersion = firmwareVersion;
      return this;
    }

    public Builder uniqueId(String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder batteryPower(int batteryPower) {
      this.batteryPower = batteryPower;
      return this;
    }

    public Builder name(String name) {
      this.name = name;
      return this;
    }

    public Builder shuffled(boolean shuffled) {
      this.shuffled = shuffled;
      return this;
    }

    public Builder timestamp(long timestamp) {
      this.timestamp = timestamp;
      return this;
    }

    public Builder password(byte[] password) {
      SDKPreconditions.checkNotNull(password, "Password is null");
      final int passwordLength = password.length;
      this.password = new byte[passwordLength];
      System.arraycopy(password, 0, this.password, 0, passwordLength);
      return this;
    }

    public EddystoneDevice build() {
      return new EddystoneDevice(this);
    }
  }
}
