package com.kontakt.sdk.android.ble.configuration;

import android.bluetooth.BluetoothAdapter;
import android.os.ParcelUuid;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;


@SuppressWarnings({"WeakerAccess", "unused"})
public class KontaktScanFilter {

    public static final KontaktScanFilter KONTAKT_SERVICE_DATA_FILTER = new KontaktScanFilter.Builder()
            .setServiceData(ParcelUuid.fromString("0000fe6a-0000-1000-8000-00805f9b34fb"), new byte[] {})
            .build();

    public static final KontaktScanFilter IBEACON_MANUFACTURER_DATA_FILTER = new KontaktScanFilter.Builder()
            .setManufacturerData(0x4c, new byte[]{})
            .build();

    public static final KontaktScanFilter KONTAKT_NAME_FILTER = new KontaktScanFilter.Builder()
            .setDeviceName("Kontakt")
            .build();

    public static final List<KontaktScanFilter> DEFAULT_FILTERS_LIST =
            Arrays.asList(KONTAKT_NAME_FILTER, KONTAKT_SERVICE_DATA_FILTER, IBEACON_MANUFACTURER_DATA_FILTER);


    private final String mDeviceName;

    private final String mDeviceAddress;

    private final ParcelUuid mServiceUuid;

    private final ParcelUuid mServiceUuidMask;

    private final ParcelUuid mServiceDataUuid;

    private final byte[] mServiceData;

    private final byte[] mServiceDataMask;

    private final int mManufacturerId;

    private final byte[] mManufacturerData;

    private final byte[] mManufacturerDataMask;

    private final EnumSet<FilterType> filterTypes;

    private KontaktScanFilter(String name, String deviceAddress, ParcelUuid uuid,
                       ParcelUuid uuidMask, ParcelUuid serviceDataUuid,
                       byte[] serviceData, byte[] serviceDataMask,
                       int manufacturerId, byte[] manufacturerData, byte[] manufacturerDataMask, EnumSet<FilterType> filterTypes) {
        mDeviceName = name;
        mServiceUuid = uuid;
        mServiceUuidMask = uuidMask;
        mDeviceAddress = deviceAddress;
        mServiceDataUuid = serviceDataUuid;
        mServiceData = serviceData;
        mServiceDataMask = serviceDataMask;
        mManufacturerId = manufacturerId;
        mManufacturerData = manufacturerData;
        mManufacturerDataMask = manufacturerDataMask;
        this.filterTypes = filterTypes;
    }
    @Override
    public String toString() {
        return "KontaktScanFilter [mDeviceName=" + mDeviceName + ", mDeviceAddress="
                + mDeviceAddress
                + ", mUuid=" + mServiceUuid + ", mUuidMask=" + mServiceUuidMask
                + ", mServiceDataUuid=" + mServiceDataUuid + ", mServiceData="
                + Arrays.toString(mServiceData) + ", mServiceDataMask="
                + Arrays.toString(mServiceDataMask) + ", mManufacturerId=" + mManufacturerId
                + ", mManufacturerData=" + Arrays.toString(mManufacturerData)
                + ", mManufacturerDataMask=" + Arrays.toString(mManufacturerDataMask) + "]";
    }

    public byte[] getManufacturerData() {
        return mManufacturerData;
    }

    public byte[] getManufacturerDataMask() {
        return mManufacturerDataMask;
    }

    public byte[] getServiceData() {
        return mServiceData;
    }

    public byte[] getServiceDataMask() {
        return mServiceDataMask;
    }

    public int getManufacturerId() {
        return mManufacturerId;
    }

    public ParcelUuid getServiceDataUuid() {
        return mServiceDataUuid;
    }

    public ParcelUuid getServiceUuid() {
        return mServiceUuid;
    }

    public ParcelUuid getServiceUuidMask() {
        return mServiceUuidMask;
    }

    public String getDeviceAddress() {
        return mDeviceAddress;
    }

    public String getDeviceName() {
        return mDeviceName;
    }

    public EnumSet<FilterType> getFilterTypes() {
        return filterTypes;
    }

    public static final class Builder {

        private String mDeviceName;
        private String mDeviceAddress;

        private ParcelUuid mServiceUuid;
        private ParcelUuid mUuidMask;

        private ParcelUuid mServiceDataUuid;
        private byte[] mServiceData;
        private byte[] mServiceDataMask;

        private int mManufacturerId = -1;
        private byte[] mManufacturerData;
        private byte[] mManufacturerDataMask;

        private EnumSet<FilterType> filterTypes = EnumSet.noneOf(FilterType.class);

        public Builder setDeviceName(String deviceName) {
            mDeviceName = deviceName;
            filterTypes.add(FilterType.DEVICE_NAME);
            return this;
        }

        public Builder setDeviceAddress(String deviceAddress) {
            if (deviceAddress != null && !BluetoothAdapter.checkBluetoothAddress(deviceAddress)) {
                throw new IllegalArgumentException("invalid device address " + deviceAddress);
            }
            mDeviceAddress = deviceAddress;
            filterTypes.add(FilterType.DEVICE_ADDRESS);
            return this;
        }

        public Builder setServiceUuid(ParcelUuid serviceUuid) {
            mServiceUuid = serviceUuid;
            mUuidMask = null;
            filterTypes.add(FilterType.SERVICE_UUID);
            return this;
        }

        public Builder setServiceUuid(ParcelUuid serviceUuid, ParcelUuid uuidMask) {
            if (mUuidMask != null && mServiceUuid == null) {
                throw new IllegalArgumentException("uuid is null while uuidMask is not null!");
            }
            mServiceUuid = serviceUuid;
            mUuidMask = uuidMask;
            filterTypes.add(FilterType.SERVICE_UUID);
            return this;
        }

        public Builder setServiceData(ParcelUuid serviceDataUuid, byte[] serviceData) {
            if (serviceDataUuid == null) {
                throw new IllegalArgumentException("serviceDataUuid is null");
            }
            mServiceDataUuid = serviceDataUuid;
            mServiceData = serviceData;
            mServiceDataMask = null;
            filterTypes.add(FilterType.SERVICE_DATA);
            return this;
        }

        public Builder setServiceData(ParcelUuid serviceDataUuid,
                                      byte[] serviceData, byte[] serviceDataMask) {
            if (serviceDataUuid == null) {
                throw new IllegalArgumentException("serviceDataUuid is null");
            }
            if (mServiceDataMask != null) {
                if (mServiceData == null) {
                    throw new IllegalArgumentException(
                            "serviceData is null while serviceDataMask is not null");
                }
                if (mServiceData.length != mServiceDataMask.length) {
                    throw new IllegalArgumentException(
                            "size mismatch for service data and service data mask");
                }
            }
            mServiceDataUuid = serviceDataUuid;
            mServiceData = serviceData;
            mServiceDataMask = serviceDataMask;
            filterTypes.add(FilterType.SERVICE_DATA);
            return this;
        }

        public Builder setManufacturerData(int manufacturerId, byte[] manufacturerData) {
            if (manufacturerData != null && manufacturerId < 0) {
                throw new IllegalArgumentException("invalid manufacture id");
            }
            mManufacturerId = manufacturerId;
            mManufacturerData = manufacturerData;
            mManufacturerDataMask = null;
            filterTypes.add(FilterType.MANUFACTURER_DATA);
            return this;
        }

        public Builder setManufacturerData(int manufacturerId, byte[] manufacturerData,
                                           byte[] manufacturerDataMask) {
            if (manufacturerData != null && manufacturerId < 0) {
                throw new IllegalArgumentException("invalid manufacture id");
            }
            if (mManufacturerDataMask != null) {
                if (mManufacturerData == null) {
                    throw new IllegalArgumentException(
                            "manufacturerData is null while manufacturerDataMask is not null");
                }

                if (mManufacturerData.length != mManufacturerDataMask.length) {
                    throw new IllegalArgumentException(
                            "size mismatch for manufacturerData and manufacturerDataMask");
                }
            }
            mManufacturerId = manufacturerId;
            mManufacturerData = manufacturerData;
            mManufacturerDataMask = manufacturerDataMask;
            filterTypes.add(FilterType.MANUFACTURER_DATA);
            return this;
        }

        public KontaktScanFilter build() {
            return new KontaktScanFilter(mDeviceName, mDeviceAddress,
                    mServiceUuid, mUuidMask,
                    mServiceDataUuid, mServiceData, mServiceDataMask,
                    mManufacturerId, mManufacturerData, mManufacturerDataMask, filterTypes);
        }
    }

    public enum FilterType {
        DEVICE_ADDRESS,
        SERVICE_DATA,
        MANUFACTURER_DATA,
        DEVICE_NAME,
        SERVICE_UUID
    }
}
