package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.List;

/**
 * Represents API response with resolved Secure Profile UIDs (from shuffled) and set of future shuffles.
 */
public class SecureProfileFutureUID implements IFutureId<SecureProfileUid> {

  public static final Creator<SecureProfileFutureUID> CREATOR = new Creator<SecureProfileFutureUID>() {
    @Override
    public SecureProfileFutureUID createFromParcel(Parcel source) {
      return new SecureProfileFutureUID(source);
    }

    @Override
    public SecureProfileFutureUID[] newArray(int size) {
      return new SecureProfileFutureUID[size];
    }
  };

  private String uniqueId;
  private SecureProfileUid queriedBy;
  private SecureProfileUid resolved;
  private List<SecureProfileUid> futureIds;

  public static Builder builder() {
    return new Builder();
  }

  SecureProfileFutureUID(Builder builder) {
    this.uniqueId = builder.uniqueId;
    this.queriedBy = builder.queriedBy;
    this.resolved = builder.resolved;
    this.futureIds = builder.futureIds;
  }

  protected SecureProfileFutureUID(Parcel in) {
    this.uniqueId = in.readString();
    this.queriedBy = in.readParcelable(SecureProfileUid.class.getClassLoader());
    this.resolved = in.readParcelable(SecureProfileUid.class.getClassLoader());
    this.futureIds = in.createTypedArrayList(SecureProfileUid.CREATOR);
  }

  public SecureProfileUid getQueriedBy() {
    return queriedBy;
  }

  public SecureProfileUid getResolved() {
    return resolved;
  }

  public List<SecureProfileUid> getFutureIds() {
    return futureIds;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof SecureProfileFutureUID)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    SecureProfileFutureUID other = (SecureProfileFutureUID) o;
    return SDKEqualsBuilder.start()
        .equals(resolved, other.resolved)
        .equals(queriedBy, other.queriedBy)
        .equals(futureIds, other.futureIds)
        .equals(uniqueId, other.uniqueId)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(resolved).append(queriedBy).append(futureIds).append(uniqueId).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.uniqueId);
    dest.writeParcelable(this.queriedBy, flags);
    dest.writeParcelable(this.resolved, flags);
    dest.writeTypedList(futureIds);
  }

  /**
   * Builder class that is used to build {@link SecureProfileFutureUID} instances from values configured by the setters.
   */
  public static class Builder {
    String uniqueId;
    SecureProfileUid queriedBy;
    SecureProfileUid resolved;
    List<SecureProfileUid> futureIds = new ArrayList<>();

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder queriedBy(final SecureProfileUid queriedBy) {
      this.queriedBy = queriedBy;
      return this;
    }

    public Builder resolved(final SecureProfileUid resolved) {
      this.resolved = resolved;
      return this;
    }

    public Builder futureIds(final List<SecureProfileUid> futureIds) {
      SDKPreconditions.checkNotNull(futureIds, "future IDs cannot be null");
      this.futureIds.clear();
      this.futureIds.addAll(futureIds);
      return this;
    }

    public SecureProfileFutureUID build() {
      return new SecureProfileFutureUID(this);
    }
  }
}
