package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import java.util.ArrayList;
import java.util.List;

/**
 * Model class representing JSON structure of PowerSaving field.
 * It holds all properties related to power saving features.
 */
public class PowerSaving implements Parcelable {

  public static final Creator<PowerSaving> CREATOR = new Creator<PowerSaving>() {
    @Override
    public PowerSaving createFromParcel(Parcel source) {
      return new PowerSaving(source);
    }

    @Override
    public PowerSaving[] newArray(int size) {
      return new PowerSaving[size];
    }
  };

  private final List<PowerSavingFeature> features;
  private final long moveSuspendTimeout;
  private final int lightSensorThreshold;
  private final int lightSensorHysteresis;
  private final long lightSensorSamplingInterval;

  PowerSaving(Builder builder) {
    this.features = builder.features;
    this.moveSuspendTimeout = builder.moveSuspendTimeout;
    this.lightSensorThreshold = builder.lightSensorThreshold;
    this.lightSensorHysteresis = builder.lightSensorHysteresis;
    this.lightSensorSamplingInterval = builder.lightSensorSamplingInterval;
  }

  protected PowerSaving(Parcel in) {
    this.features = new ArrayList<>();
    in.readList(this.features, PowerSavingFeature.class.getClassLoader());
    this.moveSuspendTimeout = in.readLong();
    this.lightSensorThreshold = in.readInt();
    this.lightSensorHysteresis = in.readInt();
    this.lightSensorSamplingInterval = in.readLong();
  }

  private PowerSaving() {
    this(new PowerSaving.Builder());
  }

  public List<PowerSavingFeature> getFeatures() {
    return features;
  }

  public long getMoveSuspendTimeout() {
    return moveSuspendTimeout;
  }

  public int getLightSensorThreshold() {
    return lightSensorThreshold;
  }

  public int getLightSensorHysteresis() {
    return lightSensorHysteresis;
  }

  public long getLightSensorSamplingInterval() {
    return lightSensorSamplingInterval;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    PowerSaving that = (PowerSaving) o;

    if (moveSuspendTimeout != that.moveSuspendTimeout) return false;
    if (lightSensorThreshold != that.lightSensorThreshold) return false;
    if (lightSensorHysteresis != that.lightSensorHysteresis) return false;
    if (lightSensorSamplingInterval != that.lightSensorSamplingInterval) return false;
    return features != null ? features.equals(that.features) : that.features == null;
  }

  @Override
  public int hashCode() {
    int result = features != null ? features.hashCode() : 0;
    result = 31 * result + (int) (moveSuspendTimeout ^ (moveSuspendTimeout >>> 32));
    result = 31 * result + lightSensorThreshold;
    result = 31 * result + lightSensorHysteresis;
    result = 31 * result + (int) (lightSensorSamplingInterval ^ (lightSensorSamplingInterval >>> 32));
    return result;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeList(this.features);
    dest.writeLong(this.moveSuspendTimeout);
    dest.writeInt(this.lightSensorThreshold);
    dest.writeInt(this.lightSensorHysteresis);
    dest.writeLong(this.lightSensorSamplingInterval);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public String toString() {
    return "PowerSaving{" +
        "features=" + features +
        ", moveSuspendTimeout=" + moveSuspendTimeout +
        ", lightSensorThreshold=" + lightSensorThreshold +
        ", lightSensorHysteresis=" + lightSensorHysteresis +
        ", lightSensorSamplingInterval=" + lightSensorSamplingInterval +
        '}';
  }

  public static class Builder {

    List<PowerSavingFeature> features = new ArrayList<>();
    long moveSuspendTimeout = -1;
    int lightSensorThreshold = -1;
    int lightSensorHysteresis = -1;
    long lightSensorSamplingInterval = -1;

    public Builder features(List<PowerSavingFeature> features) {
      this.features.clear();
      this.features.addAll(features);
      return this;
    }

    public Builder moveSuspendTimeout(long moveSuspendTimeout) {
      this.moveSuspendTimeout = moveSuspendTimeout;
      return this;
    }

    public Builder lightSensorThreshold(int lightSensorThreshold) {
      this.lightSensorThreshold = lightSensorThreshold;
      return this;
    }

    public Builder lightSensorHysteresis(int lightSensorHysteresis) {
      this.lightSensorHysteresis = lightSensorHysteresis;
      return this;
    }

    public Builder lightSensorSamplingInterval(long lightSensorSamplingInterval) {
      this.lightSensorSamplingInterval = lightSensorSamplingInterval;
      return this;
    }

    public PowerSaving build() {
      return new PowerSaving(this);
    }
  }
}
