package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Model class that represents JSON structure of Kontakt.io activity.
 * To create new instance of this class, please use {@link Activity.Builder}.
 */
public class Activity implements Parcelable {

  public static final Creator<Activity> CREATOR = new Creator<Activity>() {
    @Override
    public Activity createFromParcel(Parcel source) {
      return new Activity(source);
    }

    @Override
    public Activity[] newArray(int size) {
      return new Activity[size];
    }
  };

  private final String name;
  private final ActivityContext context;
  private final List<UUID> triggerId;
  private final UUID id;
  private final ActivityType type;

  public static Builder builder() {
    return new Builder();
  }

  Activity(Builder builder) {
    this.name = builder.name;
    this.context = builder.context;
    this.triggerId = builder.triggerId;
    this.id = builder.id;
    this.type = builder.type;
  }

  protected Activity(Parcel in) {
    this.name = in.readString();
    this.context = in.readParcelable(ActivityContext.class.getClassLoader());
    this.triggerId = new ArrayList<>();
    in.readList(this.triggerId, UUID.class.getClassLoader());
    this.id = (UUID) in.readSerializable();
    int tmpType = in.readInt();
    this.type = tmpType == -1 ? null : ActivityType.values()[tmpType];
  }

  private Activity() {
    this(new Builder());
  }

  public String getName() {
    return name;
  }

  public ActivityContext getContext() {
    return context;
  }

  public List<UUID> getTriggerIds() {
    return triggerId;
  }

  public UUID getId() {
    return id;
  }

  public ActivityType getType() {
    return type;
  }

  @Override
  public boolean equals(Object object) {
    if (object == this) {
      return true;
    }
    if (object == null || !(object instanceof Activity)) {
      return false;
    }
    final Activity trigger = (Activity) object;

    return SDKEqualsBuilder.start()
        .equals(id, trigger.id)
        .equals(name, trigger.name)
        .equals(type, trigger.type)
        .equals(context, trigger.context)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(id).append(name).append(type).append(context).build();
  }

  /**
   * Builder class that is used to build {@link Activity} instances from values configured by the setters.
   */
  public static class Builder {

    String name;
    ActivityContext context;
    List<UUID> triggerId;
    UUID id;
    ActivityType type;

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder context(final ActivityContext context) {
      this.context = context;
      return this;
    }

    public Builder triggerIds(final List<UUID> triggerIds) {
      this.triggerId = triggerIds;
      return this;
    }

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder type(final ActivityType type) {
      this.type = type;
      return this;
    }

    public Activity build() {
      return new Activity(this);
    }
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.name);
    dest.writeParcelable(this.context, flags);
    dest.writeList(this.triggerId);
    dest.writeSerializable(this.id);
    dest.writeInt(this.type == null ? -1 : this.type.ordinal());
  }
}

