package com.kontakt.sdk.android.cloud.api.executor.receivers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ReceiversApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ReceiversService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Receiver;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

/**
 * Request executor provided by {@link ReceiversApi}. Use this class if you want to create
 * receivers through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.receivers().create(receiver).execute();
 *   </code>
 * </pre>
 */
public class CreateReceiverRequestExecutor extends RequestExecutor<Receiver> {

  private final ReceiversService receiversService;
  private final Receiver receiver;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param receiversService the receivers API service.
   * @param receiver the receiver we want to create.
   */
  public CreateReceiverRequestExecutor(ReceiversService receiversService, Receiver receiver) {
    this.receiversService = receiversService;
    this.receiver = receiver;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Receiver execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<Receiver> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Receiver> prepareCall() {
    return receiversService.createReceiver(params());
  }

  private void checkPreconditions() {
    checkState(receiver.getDeviceUniqueId() != null, "cannot create receiver - specify device unique ID");
    checkState(receiver.getPlaceId() != null, "cannot create receiver - specify place ID");
    checkState(receiver.getCoordinates() != null, "cannot create receiver - specify coordinates");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, receiver.getDeviceUniqueId());
    params.put(CloudConstants.Receivers.PLACE_ID_PARAMETER, receiver.getPlaceId().toString());
    params.put(CloudConstants.Receivers.COORDINATES_PARAMETER, receiver.getCoordinates().toString());
    return params;
  }
}
